from flask import Blueprint, jsonify, request, abort
from flask_login import login_required, current_user
from app import db
from app.models import TeamMember

api_bp = Blueprint('api', __name__)

@api_bp.route('/health')
def health():
    return jsonify({'status': 'ok'})


@api_bp.route('/team-members', methods=['GET'])
@login_required
def list_team_members():
    """Get all team members for the current user"""
    members = TeamMember.query.filter_by(user_id=current_user.id).order_by(TeamMember.first_name).all()
    return jsonify({
        'team_members': [
            {
                'id': m.id,
                'first_name': m.first_name,
                'last_name': m.last_name,
                'full_name': m.full_name,
                'email': m.email,
                'years_seo_experience': m.years_seo_experience,
                'reporting_competency': m.reporting_competency
            }
            for m in members
        ]
    })


@api_bp.route('/team-members', methods=['POST'])
@login_required
def create_team_member():
    """Create a new team member"""
    data = request.get_json()
    
    if not data:
        return jsonify({'error': 'No data provided'}), 400
    
    # Validate required fields
    required = ['first_name', 'last_name', 'email']
    for field in required:
        if not data.get(field):
            return jsonify({'error': f'{field} is required'}), 400
    
    # Check if email already exists for this user's team
    existing = TeamMember.query.filter_by(
        user_id=current_user.id, 
        email=data['email']
    ).first()
    if existing:
        return jsonify({'error': 'A team member with this email already exists'}), 400
    
    # Create team member
    member = TeamMember(
        user_id=current_user.id,
        first_name=data['first_name'].strip(),
        last_name=data['last_name'].strip(),
        email=data['email'].strip().lower(),
        years_seo_experience=int(data.get('years_seo_experience', 0)),
        reporting_competency=int(data.get('reporting_competency', 5))
    )
    
    db.session.add(member)
    db.session.commit()
    
    return jsonify({
        'success': True,
        'team_member': {
            'id': member.id,
            'first_name': member.first_name,
            'last_name': member.last_name,
            'full_name': member.full_name,
            'email': member.email,
            'years_seo_experience': member.years_seo_experience,
            'reporting_competency': member.reporting_competency
        }
    }), 201


@api_bp.route('/team-members/<int:member_id>', methods=['GET'])
@login_required
def get_team_member(member_id):
    """Get a specific team member"""
    member = TeamMember.query.filter_by(id=member_id, user_id=current_user.id).first()
    if not member:
        return jsonify({'error': 'Team member not found'}), 404
    
    return jsonify({
        'id': member.id,
        'first_name': member.first_name,
        'last_name': member.last_name,
        'full_name': member.full_name,
        'email': member.email,
        'years_seo_experience': member.years_seo_experience,
        'reporting_competency': member.reporting_competency
    })

