from flask import Blueprint, render_template, send_file, abort, request, redirect, url_for, flash, jsonify
from flask_login import login_required, current_user
from app import db
from app.models import Strategy, Report, CaseStudy, StrategyActivityLog
import os
from datetime import datetime

reports_bp = Blueprint('reports', __name__)

@reports_bp.route('/')
@login_required
def index():
    # List all generated reports - Fixed: use CaseStudy.strategy instead of string 'strategy'
    reports = Report.query.join(Report.case_study).join(CaseStudy.strategy).filter(Strategy.user_id==current_user.id).order_by(Report.created_at.desc()).all()
    return render_template('reports/index.html', reports=reports)

@reports_bp.route('/strategy/<int:id>')
@login_required
def view_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    if strategy.user_id != current_user.id:
        abort(403)
    
    # Get active case study
    case_study = strategy.case_studies.filter_by(status='active').first()
    
    # Get activity logs
    activity_logs = StrategyActivityLog.query.filter_by(strategy_id=id).order_by(StrategyActivityLog.created_at.desc()).all()
    
    return render_template('reports/view_strategy.html', strategy=strategy, case_study=case_study, activity_logs=activity_logs)

@reports_bp.route('/strategy/<int:id>/edit', methods=['POST'])
@login_required
def edit_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    if strategy.user_id != current_user.id:
        abort(403)
    
    # Track changes for activity log
    changes = []
    
    old_name = strategy.name
    new_name = request.form.get('name', '').strip()
    if new_name and new_name != old_name:
        strategy.name = new_name
        changes.append(f"Name changed from '{old_name}' to '{new_name}'")
    
    old_hypothesis = strategy.hypothesis or ''
    new_hypothesis = request.form.get('hypothesis', '').strip()
    if new_hypothesis != old_hypothesis:
        strategy.hypothesis = new_hypothesis
        changes.append(f"Hypothesis updated")
    
    old_owners = strategy.owners or ''
    new_owners = request.form.get('owners', '').strip()
    if new_owners != old_owners:
        strategy.owners = new_owners
        changes.append(f"Owners changed from '{old_owners}' to '{new_owners}'")
    
    old_impl_notes = strategy.implementation_notes or ''
    new_impl_notes = request.form.get('implementation_notes', '').strip()
    if new_impl_notes != old_impl_notes:
        strategy.implementation_notes = new_impl_notes
        changes.append(f"Implementation notes updated")
    
    old_tags = strategy.tags or ''
    new_tags = request.form.get('tags', '').strip()
    if new_tags != old_tags:
        strategy.tags = new_tags
        changes.append(f"Tags changed from '{old_tags}' to '{new_tags}'")
    
    if changes:
        # Create activity log entry
        log_entry = StrategyActivityLog(
            strategy_id=strategy.id,
            action='edit',
            details='; '.join(changes),
            user_email=current_user.email
        )
        db.session.add(log_entry)
        db.session.commit()
        flash('Strategy updated successfully!', 'success')
    else:
        flash('No changes were made.', 'info')
    
    return redirect(url_for('reports.view_strategy', id=id))

@reports_bp.route('/strategy/<int:id>/delete', methods=['POST'])
@login_required
def delete_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    if strategy.user_id != current_user.id:
        abort(403)
    
    strategy_name = strategy.name
    
    # Delete related activity logs first
    StrategyActivityLog.query.filter_by(strategy_id=id).delete()
    
    # Delete related case studies and their data
    for case_study in strategy.case_studies.all():
        # Delete KPIs
        from app.models import KPI, MetricSnapshot, Upload, UploadRow
        KPI.query.filter_by(case_study_id=case_study.id).delete()
        # Delete MetricSnapshots
        MetricSnapshot.query.filter_by(case_study_id=case_study.id).delete()
        # Delete UploadRows and Uploads
        for upload in case_study.uploads.all():
            UploadRow.query.filter_by(upload_id=upload.id).delete()
        Upload.query.filter_by(case_study_id=case_study.id).delete()
        # Delete Reports
        Report.query.filter_by(case_study_id=case_study.id).delete()
        # Delete Case Study
        db.session.delete(case_study)
    
    # Delete strategy
    db.session.delete(strategy)
    db.session.commit()
    
    flash(f'Strategy "{strategy_name}" has been deleted.', 'success')
    return redirect(url_for('dashboard.index'))

@reports_bp.route('/download/<int:report_id>')
@login_required
def download(report_id):
    report = Report.query.get_or_404(report_id)
    # Check auth via joins
    if report.case_study.strategy.user_id != current_user.id:
        abort(403)
    
    if os.path.exists(report.pptx_path):
        return send_file(report.pptx_path, as_attachment=True)
    else:
        abort(404)
