import os
from flask import Blueprint, render_template, request, flash, current_app, redirect, url_for, session
from flask_login import login_required, current_user
from werkzeug.utils import secure_filename
from app import db
from app.models import OAuthToken
from app.services.oauth import get_flow, credentials_to_dict, encrypt_token, SCOPES
from datetime import datetime

settings_bp = Blueprint('settings', __name__)

@settings_bp.route('/', methods=['GET', 'POST'])
@login_required
def index():
    if request.method == 'POST':
        current_user.company_name = request.form.get('company_name')
        current_user.primary_color = request.form.get('primary_color')
        current_user.secondary_color = request.form.get('secondary_color')
        
        # Handle Logo
        if 'logo' in request.files:
            file = request.files['logo']
            if file and file.filename:
                filename = secure_filename(f"logo_{current_user.id}_{file.filename}")
                upload_path = os.path.join(current_app.config['UPLOAD_FOLDER'], 'branding')
                os.makedirs(upload_path, exist_ok=True)
                file.save(os.path.join(upload_path, filename))
                current_user.logo_path = f"uploads/branding/{filename}"
                
        db.session.commit()
        flash('Settings updated successfully', 'success')
        return redirect(url_for('settings.index'))
    
    # Check if Google is connected
    google_connected = OAuthToken.query.filter_by(user_id=current_user.id, provider='google').first() is not None
        
    return render_template('settings/index.html', google_connected=google_connected)

@settings_bp.route('/google/connect')
@login_required
def google_connect():
    """Initiate Google OAuth flow"""
    redirect_uri = url_for('settings.google_callback', _external=True)
    flow = get_flow(redirect_uri)
    
    authorization_url, state = flow.authorization_url(
        access_type='offline',
        include_granted_scopes='true',
        prompt='consent'
    )
    
    # Store state in session for verification
    session['oauth_state'] = state
    
    return redirect(authorization_url)

@settings_bp.route('/google/callback')
@login_required
def google_callback():
    """Handle Google OAuth callback"""
    # Verify state to prevent CSRF
    if request.args.get('state') != session.get('oauth_state'):
        flash('OAuth state mismatch. Please try again.', 'danger')
        return redirect(url_for('settings.index'))
    
    # Clear state from session
    session.pop('oauth_state', None)
    
    redirect_uri = url_for('settings.google_callback', _external=True)
    
    # Log the redirect URI for debugging
    current_app.logger.info(f'OAuth callback redirect_uri: {redirect_uri}')
    
    flow = get_flow(redirect_uri)
    
    try:
        flow.fetch_token(authorization_response=request.url)
    except Exception as e:
        current_app.logger.error(f'OAuth token fetch failed: {str(e)}')
        error_msg = str(e)
        if 'redirect_uri_mismatch' in error_msg.lower() or 'invalid' in error_msg.lower():
            flash(f'Google OAuth configuration error. Please ensure this redirect URI is registered in Google Cloud Console: {redirect_uri}', 'danger')
        else:
            flash(f'Failed to connect Google account: {error_msg}', 'danger')
        return redirect(url_for('settings.index'))
    
    credentials = flow.credentials
    token_dict = credentials_to_dict(credentials)
    
    # Encrypt and store
    encrypted = encrypt_token(token_dict)
    
    # Update or create token record
    token = OAuthToken.query.filter_by(user_id=current_user.id, provider='google').first()
    if token:
        token.token_blob = encrypted
        token.expires_at = credentials.expiry
        token.scope = ','.join(SCOPES)
    else:
        token = OAuthToken(
            user_id=current_user.id,
            provider='google',
            token_blob=encrypted,
            expires_at=credentials.expiry,
            scope=','.join(SCOPES)
        )
        db.session.add(token)
    
    db.session.commit()
    flash('Google account connected successfully!', 'success')
    return redirect(url_for('settings.index'))

@settings_bp.route('/google/disconnect')
@login_required
def google_disconnect():
    """Disconnect Google account"""
    OAuthToken.query.filter_by(user_id=current_user.id, provider='google').delete()
    db.session.commit()
    flash('Google account disconnected', 'info')
    return redirect(url_for('settings.index'))
