from datetime import datetime
from dateutil.relativedelta import relativedelta
from flask import Blueprint, render_template, request, redirect, url_for, flash
from flask_login import login_required, current_user
from app import db
from app.models import Strategy, CaseStudy, KPI, StrategyActivityLog, TeamMember

wizard_bp = Blueprint('wizard', __name__)

def generate_unique_strategy_name(user_id, base_name="New Strategy"):
    """Generate a unique strategy name by appending a number if needed."""
    existing_names = set(
        s.name for s in Strategy.query.filter_by(user_id=user_id).all()
    )
    
    if base_name not in existing_names:
        return base_name
    
    # Find the next available number
    counter = 2
    while f"{base_name} {counter}" in existing_names:
        counter += 1
    
    return f"{base_name} {counter}"

@wizard_bp.route('/new')
@login_required
def new():
    # Initialize a new draft strategy with unique name
    unique_name = generate_unique_strategy_name(current_user.id, "New Strategy")
    strategy = Strategy(user=current_user, name=unique_name, created_at=datetime.utcnow())
    db.session.add(strategy)
    db.session.commit() # Get ID
    
    # Log strategy creation
    log_entry = StrategyActivityLog(
        strategy_id=strategy.id,
        action='create',
        details=f'Strategy "{unique_name}" created',
        user_email=current_user.email
    )
    db.session.add(log_entry)
    
    case_study = CaseStudy(strategy=strategy, status='draft')
    db.session.add(case_study)
    db.session.commit()
    
    return redirect(url_for('wizard.step', step=1, id=case_study.id))

@wizard_bp.route('/<int:id>/step/<int:step>', methods=['GET', 'POST'])
@login_required
def step(id, step):
    case_study = CaseStudy.query.get_or_404(id)
    strategy = case_study.strategy
    
    if strategy.user_id != current_user.id:
        flash('Unauthorized', 'danger')
        return redirect(url_for('dashboard.index'))
    
    # Handle Posts for each step
    if request.method == 'POST':
        if step == 1:
            # Ensure unique strategy name
            new_name = request.form['name'].strip()
            if new_name != strategy.name:
                # Check if name already exists for this user
                existing = Strategy.query.filter(
                    Strategy.user_id == current_user.id,
                    Strategy.name == new_name,
                    Strategy.id != strategy.id
                ).first()
                if existing:
                    new_name = generate_unique_strategy_name(current_user.id, new_name)
                    flash(f'Strategy name already exists. Changed to "{new_name}"', 'info')
            
            strategy.name = new_name
            strategy.hypothesis = request.form['hypothesis']
            
            # Handle owner selection
            owner_id = request.form.get('owner_id', '')
            if owner_id and owner_id.isdigit():
                member = TeamMember.query.filter_by(id=int(owner_id), user_id=current_user.id).first()
                if member:
                    strategy.owners = member.full_name
            else:
                strategy.owners = request.form.get('owners', '')
            
            db.session.commit()
            return redirect(url_for('wizard.step', id=id, step=2))
        
        elif step == 2:
            # Only collect test period dates - benchmark is auto-calculated
            post_start = datetime.strptime(request.form['post_start'], '%Y-%m-%d').date()
            post_end = datetime.strptime(request.form['post_end'], '%Y-%m-%d').date()
            
            # Calculate benchmark period: same duration, 1 year prior
            # The benchmark period is the same time chunk but 1 year earlier
            benchmark_start = post_start - relativedelta(years=1)
            benchmark_end = post_end - relativedelta(years=1)
            
            case_study.post_start_date = post_start
            case_study.post_end_date = post_end
            case_study.benchmark_start_date = benchmark_start
            case_study.benchmark_end_date = benchmark_end
            db.session.commit()
            return redirect(url_for('wizard.step', id=id, step=3))
            
        elif step == 3:
            # Save selected properties
            case_study.ga4_property_id = request.form.get('ga4_property')
            case_study.gsc_site_url = request.form.get('gsc_property')
            db.session.commit()
            return redirect(url_for('wizard.step', id=id, step=4))
            
        elif step == 4:
            # Save KPIs
            # Clear existing
            KPI.query.filter_by(case_study_id=id).delete()
            
            metrics = request.form.getlist('metrics')
            for m in metrics:
                source, name = m.split(':')
                kpi = KPI(case_study=case_study, source=source, metric=name)
                db.session.add(kpi)
            db.session.commit()
            return redirect(url_for('wizard.step', id=id, step=6)) # Skipping upload for now to keep quick
            
        elif step == 6:
            case_study.status = 'active'
            
            # Log activation
            log_entry = StrategyActivityLog(
                strategy_id=strategy.id,
                action='activate',
                details='Strategy activated and data fetching started',
                user_email=current_user.email
            )
            db.session.add(log_entry)
            db.session.commit()
            
            # Trigger Backgroud Job Here
            from app.tasks import fetch_data_task
            fetch_data_task.delay(case_study.id)
            
            flash('Strategy created! Data fetching has started.', 'success')
            return redirect(url_for('dashboard.index'))

    # Build extra context for specific steps
    extra_context = {}
    
    if step == 1:
        # Fetch team members for owner dropdown
        team_members = TeamMember.query.filter_by(user_id=current_user.id).order_by(TeamMember.first_name).all()
        extra_context['team_members'] = team_members
    
    elif step == 3:
        # Fetch real GA4 and GSC properties from the user's linked Google account
        from app.services.ga4 import get_ga4_properties, get_gsc_sites
        from app.models import OAuthToken
        
        # Check if Google is connected
        google_connected = OAuthToken.query.filter_by(user_id=current_user.id, provider='google').first() is not None
        extra_context['google_connected'] = google_connected
        
        if google_connected:
            extra_context['ga4_properties'] = get_ga4_properties(current_user.id)
            extra_context['gsc_sites'] = get_gsc_sites(current_user.id)
        else:
            extra_context['ga4_properties'] = []
            extra_context['gsc_sites'] = []

    progress = int((step / 6) * 100)
    return render_template(
        f'wizard/step_{step}.html', 
        case_study=case_study, 
        strategy=strategy, 
        step=step, 
        progress=progress,
        **extra_context
    )


