from datetime import datetime
from flask_login import UserMixin
from werkzeug.security import generate_password_hash, check_password_hash
from app import db

class User(UserMixin, db.Model):
    __tablename__ = 'users'
    
    id = db.Column(db.Integer, primary_key=True)
    email = db.Column(db.String(120), unique=True, index=True)
    password_hash = db.Column(db.String(256))
    
    # Branding
    company_name = db.Column(db.String(128))
    logo_path = db.Column(db.String(256))
    primary_color = db.Column(db.String(7), default='#0d6efd') # Bootstrap primary
    secondary_color = db.Column(db.String(7), default='#6c757d') # Bootstrap secondary
    
    strategies = db.relationship('Strategy', backref='user', lazy='dynamic')
    oauth_tokens = db.relationship('OAuthToken', backref='user', lazy='dynamic')

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

class OAuthToken(db.Model):
    __tablename__ = 'oauth_tokens'
    
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey('users.id'))
    provider = db.Column(db.String(50)) # google
    token_blob = db.Column(db.LargeBinary) # Encrypted
    expires_at = db.Column(db.DateTime)
    scope = db.Column(db.Text)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)


class TeamMember(db.Model):
    """Team members who can be assigned as strategy owners"""
    __tablename__ = 'team_members'
    
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)  # Parent user who created this team member
    first_name = db.Column(db.String(64), nullable=False)
    last_name = db.Column(db.String(64), nullable=False)
    email = db.Column(db.String(120), nullable=False)
    years_seo_experience = db.Column(db.Integer, default=0)
    reporting_competency = db.Column(db.Integer, default=5)  # 1-10 star rating
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    # Relationship to parent user
    created_by = db.relationship('User', backref=db.backref('team_members', lazy='dynamic'))
    
    @property
    def full_name(self):
        return f"{self.first_name} {self.last_name}"
    
    def __repr__(self):
        return f'<TeamMember {self.full_name}>'
