"""
Google Analytics 4 and Search Console API Services
"""
from flask import current_app
from google.oauth2.credentials import Credentials
from googleapiclient.discovery import build
from googleapiclient.errors import HttpError
from app.models import OAuthToken
from app.services.oauth import decrypt_token, get_credentials_from_token
import logging

logger = logging.getLogger(__name__)


def get_user_credentials(user_id):
    """Get valid Google credentials for a user"""
    token = OAuthToken.query.filter_by(user_id=user_id, provider='google').first()
    if not token:
        return None
    
    try:
        token_dict = decrypt_token(token.token_blob)
        if not token_dict:
            return None
        credentials = get_credentials_from_token(token_dict)
        return credentials
    except Exception as e:
        logger.error(f"Failed to get credentials for user {user_id}: {e}")
        return None


def get_ga4_properties(user_id):
    """
    Fetch all GA4 properties accessible by the user.
    Returns a list of dicts with 'id' and 'name' keys.
    """
    credentials = get_user_credentials(user_id)
    if not credentials:
        logger.warning(f"No credentials found for user {user_id}")
        return []
    
    try:
        # Use GA4 Admin API to list properties
        service = build('analyticsadmin', 'v1beta', credentials=credentials)
        
        # First get account summaries (includes properties)
        accounts_response = service.accountSummaries().list().execute()
        
        properties = []
        account_summaries = accounts_response.get('accountSummaries', [])
        
        for account in account_summaries:
            account_name = account.get('displayName', 'Unknown Account')
            for prop in account.get('propertySummaries', []):
                # Property name format: properties/123456789
                prop_resource = prop.get('property', '')
                prop_id = prop_resource.split('/')[-1] if prop_resource else ''
                prop_display_name = prop.get('displayName', 'Unnamed Property')
                
                properties.append({
                    'id': prop_id,
                    'name': f"{prop_display_name} ({account_name})",
                    'resource_name': prop_resource
                })
        
        logger.info(f"Found {len(properties)} GA4 properties for user {user_id}")
        return properties
        
    except HttpError as e:
        logger.error(f"GA4 API error for user {user_id}: {e}")
        return []
    except Exception as e:
        logger.error(f"Unexpected error fetching GA4 properties for user {user_id}: {e}")
        return []


def get_gsc_sites(user_id):
    """
    Fetch all Search Console sites accessible by the user.
    Returns a list of dicts with 'url' and 'permission_level' keys.
    """
    credentials = get_user_credentials(user_id)
    if not credentials:
        logger.warning(f"No credentials found for user {user_id}")
        return []
    
    try:
        # Use Search Console API to list sites
        service = build('searchconsole', 'v1', credentials=credentials)
        
        response = service.sites().list().execute()
        sites = []
        
        for site in response.get('siteEntry', []):
            site_url = site.get('siteUrl', '')
            permission = site.get('permissionLevel', 'unknown')
            
            # Format display name
            display_name = site_url
            if site_url.startswith('sc-domain:'):
                display_name = f"🌐 {site_url}"  # Domain property
            else:
                display_name = f"📄 {site_url}"  # URL prefix
            
            sites.append({
                'url': site_url,
                'display_name': display_name,
                'permission': permission
            })
        
        logger.info(f"Found {len(sites)} GSC sites for user {user_id}")
        return sites
        
    except HttpError as e:
        logger.error(f"GSC API error for user {user_id}: {e}")
        return []
    except Exception as e:
        logger.error(f"Unexpected error fetching GSC sites for user {user_id}: {e}")
        return []


class GA4Service:
    """Service for fetching GA4 reports"""
    
    def __init__(self, user):
        self.user = user
        self.credentials = get_user_credentials(user.id)
        
    def fetch_report(self, property_id, start_date, end_date, metrics, dimensions=None):
        """Fetch a GA4 report for the given parameters"""
        if not self.credentials:
            return {'error': 'No Google credentials found'}
        
        try:
            service = build('analyticsdata', 'v1beta', credentials=self.credentials)
            
            request_body = {
                'dateRanges': [{'startDate': start_date, 'endDate': end_date}],
                'metrics': [{'name': m} for m in metrics]
            }
            
            if dimensions:
                request_body['dimensions'] = [{'name': d} for d in dimensions]
            
            response = service.properties().runReport(
                property=f'properties/{property_id}',
                body=request_body
            ).execute()
            
            return response
            
        except HttpError as e:
            logger.error(f"GA4 report error: {e}")
            return {'error': str(e)}
        except Exception as e:
            logger.error(f"Unexpected error in GA4 report: {e}")
            return {'error': str(e)}


class GSCService:
    """Service for fetching Search Console data"""
    
    def __init__(self, user):
        self.user = user
        self.credentials = get_user_credentials(user.id)
        
    def fetch_search_analytics(self, site_url, start_date, end_date, dimensions=None):
        """Fetch Search Console analytics for the given parameters"""
        if not self.credentials:
            return {'error': 'No Google credentials found'}
        
        try:
            service = build('searchconsole', 'v1', credentials=self.credentials)
            
            request_body = {
                'startDate': start_date,
                'endDate': end_date,
                'dimensions': dimensions or ['query', 'page'],
                'rowLimit': 1000
            }
            
            response = service.searchanalytics().query(
                siteUrl=site_url,
                body=request_body
            ).execute()
            
            return response
            
        except HttpError as e:
            logger.error(f"GSC analytics error: {e}")
            return {'error': str(e)}
        except Exception as e:
            logger.error(f"Unexpected error in GSC analytics: {e}")
            return {'error': str(e)}
