import os
import json
from datetime import datetime, timedelta
from flask import current_app, url_for, session
from google_auth_oauthlib.flow import Flow
from google.oauth2.credentials import Credentials
from google.auth.transport.requests import Request
from cryptography.fernet import Fernet

# Scopes required for GA4 and GSC
SCOPES = [
    'https://www.googleapis.com/auth/analytics.readonly',
    'https://www.googleapis.com/auth/webmasters.readonly',
    'openid',
    'https://www.googleapis.com/auth/userinfo.email'
]

def get_flow(redirect_uri):
    """Create OAuth flow for Google authentication"""
    client_config = {
        "web": {
            "client_id": current_app.config['GOOGLE_CLIENT_ID'],
            "client_secret": current_app.config['GOOGLE_CLIENT_SECRET'],
            "auth_uri": "https://accounts.google.com/o/oauth2/auth",
            "token_uri": "https://oauth2.googleapis.com/token",
            "redirect_uris": [redirect_uri]
        }
    }
    
    flow = Flow.from_client_config(
        client_config,
        scopes=SCOPES
    )
    flow.redirect_uri = redirect_uri
    return flow

def credentials_to_dict(credentials):
    """Convert credentials to a dictionary for storage"""
    return {
        'token': credentials.token,
        'refresh_token': credentials.refresh_token,
        'token_uri': credentials.token_uri,
        'client_id': credentials.client_id,
        'client_secret': credentials.client_secret,
        'scopes': list(credentials.scopes),
        'expiry': credentials.expiry.isoformat() if credentials.expiry else None
    }

def encrypt_token(token_dict):
    """Encrypt token data for storage"""
    key = current_app.config.get('ENCRYPTION_KEY')
    if not key or key == 'generate-fernet-key-here':
        # For dev, use a default key (insecure but functional)
        key = Fernet.generate_key()
    elif isinstance(key, str):
        key = key.encode()
    
    cipher = Fernet(key)
    return cipher.encrypt(json.dumps(token_dict).encode())

def decrypt_token(encrypted_blob):
    """Decrypt token data from storage"""
    key = current_app.config.get('ENCRYPTION_KEY')
    if not key or key == 'generate-fernet-key-here':
        return None
    elif isinstance(key, str):
        key = key.encode()
    
    cipher = Fernet(key)
    return json.loads(cipher.decrypt(encrypted_blob).decode())

def get_credentials_from_token(token_dict):
    """Recreate credentials from stored dictionary"""
    expiry = None
    if token_dict.get('expiry'):
        expiry = datetime.fromisoformat(token_dict['expiry'])
    
    return Credentials(
        token=token_dict['token'],
        refresh_token=token_dict.get('refresh_token'),
        token_uri=token_dict['token_uri'],
        client_id=token_dict['client_id'],
        client_secret=token_dict['client_secret'],
        scopes=token_dict['scopes'],
        expiry=expiry
    )
