import React, { useState, useEffect } from 'react';
import { Download, ChevronDown, ChevronUp, Calendar, TrendingUp, TrendingDown, ArrowUpDown } from 'lucide-react';
import axios from 'axios';

const DataImportPanel = ({ strategyId }) => {
    const [isExpanded, setIsExpanded] = useState(false);
    const [activeTab, setActiveTab] = useState('GSC');
    const [isImporting, setIsImporting] = useState(false);
    const [importProgress, setImportProgress] = useState(0);
    const [gscData, setGscData] = useState([]);
    const [ga4Data, setGa4Data] = useState([]);
    const [dateRange, setDateRange] = useState({
        start: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
        end: new Date().toISOString().split('T')[0]
    });
    const [sortConfig, setSortConfig] = useState({ key: 'date', direction: 'desc' });
    const [benchmarkDates, setBenchmarkDates] = useState({
        start: '',
        end: ''
    });

    const handleImportData = async () => {
        setIsImporting(true);
        setImportProgress(0);

        try {
            // Simulate progress
            const progressInterval = setInterval(() => {
                setImportProgress(prev => {
                    if (prev >= 90) {
                        clearInterval(progressInterval);
                        return 90;
                    }
                    return prev + 10;
                });
            }, 300);

            // Import GSC data
            const gscResponse = await axios.post(`/api/strategies/${strategyId}/import-data`, {
                platform: 'GSC',
                dateRange
            });

            // Import GA4 data
            const ga4Response = await axios.post(`/api/strategies/${strategyId}/import-data`, {
                platform: 'GA4',
                dateRange
            });

            clearInterval(progressInterval);
            setImportProgress(100);

            setGscData(gscResponse.data.data || []);
            setGa4Data(ga4Response.data.data || []);

            // Auto-expand after successful import
            setIsExpanded(true);

            setTimeout(() => {
                setIsImporting(false);
                setImportProgress(0);
            }, 1000);

        } catch (error) {
            console.error('Import failed:', error);
            alert('Failed to import data. Please ensure accounts are connected.');
            setIsImporting(false);
            setImportProgress(0);
        }
    };

    const handleSort = (key) => {
        setSortConfig({
            key,
            direction: sortConfig.key === key && sortConfig.direction === 'asc' ? 'desc' : 'asc'
        });
    };

    const sortData = (data) => {
        if (!sortConfig.key) return data;

        return [...data].sort((a, b) => {
            const aVal = a[sortConfig.key];
            const bVal = b[sortConfig.key];

            if (aVal === null || aVal === undefined) return 1;
            if (bVal === null || bVal === undefined) return -1;

            const comparison = aVal > bVal ? 1 : aVal < bVal ? -1 : 0;
            return sortConfig.direction === 'asc' ? comparison : -comparison;
        });
    };

    const filterByBenchmark = (data) => {
        if (!benchmarkDates.start || !benchmarkDates.end) return data;

        return data.filter(item => {
            const itemDate = new Date(item.date);
            const startDate = new Date(benchmarkDates.start);
            const endDate = new Date(benchmarkDates.end);
            return itemDate >= startDate && itemDate <= endDate;
        });
    };

    const SortIcon = ({ columnKey }) => {
        if (sortConfig.key !== columnKey) {
            return <ArrowUpDown size={14} className="text-gray-500 ml-1 inline" />;
        }
        return sortConfig.direction === 'asc'
            ? <TrendingUp size={14} className="text-blue-400 ml-1 inline" />
            : <TrendingDown size={14} className="text-blue-400 ml-1 inline" />;
    };

    const renderGSCTable = () => {
        const processedData = sortData(filterByBenchmark(gscData));

        return (
            <div className="overflow-x-auto">
                <table className="w-full text-sm">
                    <thead className="bg-[rgba(0,0,0,0.3)] border-b border-[var(--color-border)]">
                        <tr>
                            <th className="px-4 py-3 text-left cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('date')}>
                                Date <SortIcon columnKey="date" />
                            </th>
                            <th className="px-4 py-3 text-left cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('url')}>
                                URL <SortIcon columnKey="url" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('clicks')}>
                                Clicks <SortIcon columnKey="clicks" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('impressions')}>
                                Impressions <SortIcon columnKey="impressions" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('ctr')}>
                                CTR <SortIcon columnKey="ctr" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('position')}>
                                Position <SortIcon columnKey="position" />
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        {processedData.length > 0 ? (
                            processedData.map((row, idx) => (
                                <tr key={idx} className="border-b border-[var(--color-border)] hover:bg-[rgba(255,255,255,0.02)]">
                                    <td className="px-4 py-2 text-gray-300">{new Date(row.date).toLocaleDateString()}</td>
                                    <td className="px-4 py-2 text-blue-300 truncate max-w-xs" title={row.url}>{row.url}</td>
                                    <td className="px-4 py-2 text-right text-green-400 font-mono">{row.clicks?.toLocaleString() || 0}</td>
                                    <td className="px-4 py-2 text-right text-purple-400 font-mono">{row.impressions?.toLocaleString() || 0}</td>
                                    <td className="px-4 py-2 text-right text-yellow-400 font-mono">{(row.ctr * 100).toFixed(2)}%</td>
                                    <td className="px-4 py-2 text-right text-orange-400 font-mono">{row.position?.toFixed(1) || 'N/A'}</td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan="6" className="px-4 py-8 text-center text-gray-500 italic">
                                    No data available. Click "Import Data" to fetch GSC metrics.
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>
        );
    };

    const renderGA4Table = () => {
        const processedData = sortData(filterByBenchmark(ga4Data));

        return (
            <div className="overflow-x-auto">
                <table className="w-full text-sm">
                    <thead className="bg-[rgba(0,0,0,0.3)] border-b border-[var(--color-border)]">
                        <tr>
                            <th className="px-4 py-3 text-left cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('date')}>
                                Date <SortIcon columnKey="date" />
                            </th>
                            <th className="px-4 py-3 text-left cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('url')}>
                                URL <SortIcon columnKey="url" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('sessions')}>
                                Sessions <SortIcon columnKey="sessions" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('pageviews')}>
                                Pageviews <SortIcon columnKey="pageviews" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('bounceRate')}>
                                Bounce Rate <SortIcon columnKey="bounceRate" />
                            </th>
                            <th className="px-4 py-3 text-right cursor-pointer hover:bg-[rgba(255,255,255,0.05)]" onClick={() => handleSort('avgSessionDuration')}>
                                Avg Duration <SortIcon columnKey="avgSessionDuration" />
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        {processedData.length > 0 ? (
                            processedData.map((row, idx) => (
                                <tr key={idx} className="border-b border-[var(--color-border)] hover:bg-[rgba(255,255,255,0.02)]">
                                    <td className="px-4 py-2 text-gray-300">{new Date(row.date).toLocaleDateString()}</td>
                                    <td className="px-4 py-2 text-blue-300 truncate max-w-xs" title={row.url}>{row.url}</td>
                                    <td className="px-4 py-2 text-right text-green-400 font-mono">{row.sessions?.toLocaleString() || 0}</td>
                                    <td className="px-4 py-2 text-right text-purple-400 font-mono">{row.pageviews?.toLocaleString() || 0}</td>
                                    <td className="px-4 py-2 text-right text-red-400 font-mono">{(row.bounceRate * 100).toFixed(2)}%</td>
                                    <td className="px-4 py-2 text-right text-yellow-400 font-mono">{row.avgSessionDuration ? `${Math.floor(row.avgSessionDuration / 60)}:${(row.avgSessionDuration % 60).toFixed(0).padStart(2, '0')}` : 'N/A'}</td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan="6" className="px-4 py-8 text-center text-gray-500 italic">
                                    No data available. Click "Import Data" to fetch GA4 metrics.
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>
        );
    };

    return (
        <div className="space-y-4">
            {/* Import Button and Progress */}
            <div className="flex items-center gap-4">
                <button
                    onClick={handleImportData}
                    disabled={isImporting}
                    className="flex items-center gap-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white px-4 py-2 rounded-lg transition-all shadow-lg disabled:opacity-50 disabled:cursor-not-allowed"
                >
                    <Download size={18} />
                    {isImporting ? 'Importing...' : 'Import Data'}
                </button>

                {isImporting && (
                    <div className="flex-1 max-w-md">
                        <div className="relative h-2 bg-[rgba(0,0,0,0.3)] rounded-full overflow-hidden">
                            <div
                                className="absolute top-0 left-0 h-full bg-gradient-to-r from-blue-500 to-purple-500 transition-all duration-300 ease-out"
                                style={{ width: `${importProgress}%` }}
                            />
                        </div>
                        <p className="text-xs text-gray-400 mt-1">{importProgress}% complete</p>
                    </div>
                )}
            </div>

            {/* Collapsible Data Panel */}
            {(gscData.length > 0 || ga4Data.length > 0) && (
                <div className="glass-card overflow-hidden">
                    {/* Header */}
                    <button
                        onClick={() => setIsExpanded(!isExpanded)}
                        className="w-full px-6 py-4 flex items-center justify-between hover:bg-[rgba(255,255,255,0.02)] transition-colors"
                    >
                        <div className="flex items-center gap-3">
                            <Download size={20} className="text-blue-400" />
                            <h3 className="text-lg font-semibold">Imported Benchmark Data</h3>
                            <span className="text-xs text-gray-500 bg-[rgba(255,255,255,0.05)] px-2 py-1 rounded">
                                {gscData.length + ga4Data.length} records
                            </span>
                        </div>
                        {isExpanded ? <ChevronUp size={20} /> : <ChevronDown size={20} />}
                    </button>

                    {/* Expanded Content */}
                    {isExpanded && (
                        <div className="border-t border-[var(--color-border)] p-6 space-y-4">
                            {/* Date Range Filter for Benchmark */}
                            <div className="flex items-center gap-4 pb-4 border-b border-[var(--color-border)]">
                                <Calendar size={18} className="text-purple-400" />
                                <span className="text-sm font-medium">Benchmark Date Range:</span>
                                <input
                                    type="date"
                                    value={benchmarkDates.start}
                                    onChange={(e) => setBenchmarkDates({ ...benchmarkDates, start: e.target.value })}
                                    className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-1 text-sm text-white"
                                />
                                <span className="text-gray-500">to</span>
                                <input
                                    type="date"
                                    value={benchmarkDates.end}
                                    onChange={(e) => setBenchmarkDates({ ...benchmarkDates, end: e.target.value })}
                                    className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-1 text-sm text-white"
                                />
                                {benchmarkDates.start && benchmarkDates.end && (
                                    <button
                                        onClick={() => setBenchmarkDates({ start: '', end: '' })}
                                        className="text-xs text-red-400 hover:text-red-300"
                                    >
                                        Clear
                                    </button>
                                )}
                            </div>

                            {/* Tabs */}
                            <div className="flex gap-2 border-b border-[var(--color-border)]">
                                <button
                                    onClick={() => setActiveTab('GSC')}
                                    className={`px-6 py-2 font-medium transition-all relative ${activeTab === 'GSC'
                                            ? 'text-blue-400'
                                            : 'text-gray-400 hover:text-gray-300'
                                        }`}
                                >
                                    Google Search Console
                                    {activeTab === 'GSC' && (
                                        <div className="absolute bottom-0 left-0 right-0 h-0.5 bg-blue-400" />
                                    )}
                                </button>
                                <button
                                    onClick={() => setActiveTab('GA4')}
                                    className={`px-6 py-2 font-medium transition-all relative ${activeTab === 'GA4'
                                            ? 'text-purple-400'
                                            : 'text-gray-400 hover:text-gray-300'
                                        }`}
                                >
                                    Google Analytics 4
                                    {activeTab === 'GA4' && (
                                        <div className="absolute bottom-0 left-0 right-0 h-0.5 bg-purple-400" />
                                    )}
                                </button>
                            </div>

                            {/* Table Content */}
                            <div className="mt-4">
                                {activeTab === 'GSC' ? renderGSCTable() : renderGA4Table()}
                            </div>
                        </div>
                    )}
                </div>
            )}
        </div>
    );
};

export default DataImportPanel;
