import React, { useState } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { LayoutDashboard, Settings, PlusCircle, LineChart, Lightbulb, Activity, Calendar, UserCog, FileText } from 'lucide-react';

const Layout = ({ children }) => {
    const location = useLocation();
    const [showUserMenu, setShowUserMenu] = useState(false);

    const navItems = [
        {
            section: 'Strategies',
            items: [
                { icon: Lightbulb, label: 'All Strategies', path: '/' },
                { icon: Activity, label: 'Live / Measuring', path: '/strategies/live' },
                { icon: Calendar, label: 'Planning', path: '/strategies/planning' },
            ]
        },
        {
            section: 'Reports',
            items: [
                { icon: FileText, label: 'Performance Reports', path: '/performance-reports' },
            ]
        },
        {
            section: 'Settings',
            items: [
                { icon: UserCog, label: 'Authorized Accounts', path: '/settings/accounts' },
            ]
        }
    ];

    return (
        <div className="flex h-screen bg-[var(--color-bg)] text-[var(--color-text-main)] overflow-hidden">
            {/* Sidebar */}
            <aside className="w-64 glass-panel flex flex-col border-r border-[var(--color-border)]">
                <div className="p-6">
                    <h1 className="text-xl font-bold bg-gradient-to-r from-[var(--color-primary)] to-[var(--color-secondary)] bg-clip-text text-transparent">
                        SEO Tracker V2
                    </h1>
                </div>

                <nav className="flex-1 px-4 space-y-6 overflow-y-auto">
                    {navItems.map((section, idx) => (
                        <div key={idx}>
                            <h3 className="text-xs font-semibold text-[var(--color-text-muted)] uppercase tracking-wider mb-2 px-2">
                                {section.section}
                            </h3>
                            <div className="space-y-1">
                                {section.items.map((item) => (
                                    <Link
                                        key={item.path}
                                        to={item.path}
                                        className={`flex items-center gap-3 px-3 py-2 rounded-[var(--radius-md)] transition-all duration-200 ${location.pathname === item.path
                                            ? 'bg-[var(--color-primary-glow)] text-white shadow-[var(--shadow-glow)]'
                                            : 'text-[var(--color-text-muted)] hover:bg-[var(--color-bg-card)] hover:text-white'
                                            }`}
                                    >
                                        <item.icon size={18} />
                                        <span className="font-medium text-sm">{item.label}</span>
                                    </Link>
                                ))}
                            </div>
                        </div>
                    ))}
                </nav>

                <div className="p-4 border-t border-[var(--color-border)] relative">
                    <button
                        onClick={() => setShowUserMenu(!showUserMenu)}
                        className="flex items-center gap-3 w-full hover:bg-[rgba(255,255,255,0.05)] p-2 rounded-[var(--radius-md)] transition-colors text-left"
                    >
                        <div className="w-8 h-8 rounded-full bg-gradient-to-br from-[var(--color-primary)] to-[var(--color-secondary)]"></div>
                        <div className="flex-1">
                            <p className="text-sm font-medium">User Admin</p>
                            <p className="text-xs text-[var(--color-text-muted)]">admin@jetlife.com</p>
                        </div>
                    </button>

                    {showUserMenu && (
                        <div className="absolute bottom-full left-4 right-4 bg-[#1e293b] border border-[var(--color-border)] rounded-[var(--radius-md)] p-1 mb-2 shadow-xl z-50">
                            <Link
                                to="/settings"
                                className="flex items-center gap-2 px-3 py-2 text-sm hover:bg-[rgba(255,255,255,0.1)] rounded-[var(--radius-sm)] transition-colors text-white"
                                onClick={() => setShowUserMenu(false)}
                            >
                                <Settings size={16} /> Settings
                            </Link>
                            <button className="flex items-center gap-2 px-3 py-2 text-sm text-red-400 hover:bg-[rgba(248,113,113,0.1)] rounded-[var(--radius-sm)] w-full text-left transition-colors">
                                <span className="w-4 h-4 rounded-full border border-current flex items-center justify-center text-[10px]">?</span> Sign Out
                            </button>
                        </div>
                    )}
                </div>
            </aside>

            {/* Main Content */}
            <main className="flex-1 overflow-auto relative">
                <div className="absolute inset-0 pointer-events-none bg-[radial-gradient(ellipse_at_top_right,_var(--tw-gradient-stops))] from-[rgba(59,130,246,0.1)] via-transparent to-transparent"></div>
                <div className="p-8 relative z-10">
                    {children}
                </div>
            </main>
        </div>
    );
};

export default Layout;
