import React, { useState, useEffect } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Plus, ArrowRight, Activity, Calendar } from 'lucide-react';
import { motion } from 'framer-motion';

const Dashboard = ({ filter }) => {
    const navigate = useNavigate();
    const [strategies, setStrategies] = useState([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        fetchStrategies();
    }, [filter]);

    const fetchStrategies = async () => {
        setLoading(true);
        try {
            const res = await axios.get('/api/strategies');
            // Basic client-side filtering placeholder
            let data = res.data;
            if (filter === 'live') {
                // filter logic if we had status
            }
            setStrategies(data);
        } catch (err) {
            console.error("Failed to fetch strategies", err);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="space-y-8">
            <div className="flex justify-between items-center">
                <div>
                    <h2 className="text-3xl font-bold text-white mb-2">
                        {filter === 'live' ? 'Live Strategies' : filter === 'planning' ? 'Planning Phase' : 'All Strategies'}
                    </h2>
                    <p className="text-[var(--color-text-muted)]">Track and manage your SEO campaigns</p>
                </div>
                <button
                    onClick={() => navigate('/strategy/new')}
                    className="flex items-center gap-2 bg-[var(--color-primary)] hover:bg-blue-600 text-white px-4 py-2 rounded-[var(--radius-md)] transition-colors shadow-[var(--shadow-glow)]"
                >
                    <Plus size={20} />
                    <span>New Strategy</span>
                </button>
            </div>

            {loading ? (
                <div className="text-center py-20 text-[var(--color-text-muted)]">Loading strategies...</div>
            ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {strategies.map((strategy) => (
                        <motion.div
                            key={strategy.id}
                            initial={{ opacity: 0, scale: 0.95 }}
                            animate={{ opacity: 1, scale: 1 }}
                            whileHover={{ y: -5 }}
                            onClick={() => navigate(`/strategy/${strategy.id}`)}
                            className="glass-card p-6 flex flex-col justify-between h-56 hover:shadow-[var(--shadow-glow)] transition-all cursor-pointer group"
                        >
                            <div>
                                <div className="flex justify-between items-start mb-4">
                                    <div className="h-10 w-10 rounded-full bg-[rgba(59,130,246,0.2)] flex items-center justify-center text-[var(--color-primary)]">
                                        <Activity size={20} />
                                    </div>
                                    <span className="text-xs px-2 py-1 rounded-full bg-[rgba(255,255,255,0.1)] text-[var(--color-text-muted)]">
                                        {strategy.update_frequency}
                                    </span>
                                </div>
                                <h3 className="text-xl font-semibold mb-2 group-hover:text-[var(--color-primary)] transition-colors">{strategy.name}</h3>
                                <p className="text-sm text-[var(--color-text-muted)] line-clamp-2">{strategy.description}</p>
                            </div>

                            <div className="flex justify-between items-center mt-4 pt-4 border-t border-[rgba(255,255,255,0.05)]">
                                <span className="text-xs text-[var(--color-text-muted)] flex items-center gap-1">
                                    <Calendar size={12} />
                                    {new Date(strategy.created_at).toLocaleDateString()}
                                </span>

                                <div className="text-sm font-medium text-[var(--color-primary)] flex items-center gap-1 hover:gap-2 transition-all">
                                    View Details <ArrowRight size={16} />
                                </div>
                            </div>
                        </motion.div>
                    ))}

                    {strategies.length === 0 && !loading && (
                        <div className="col-span-full text-center py-20 glass-panel rounded-[var(--radius-lg)]">
                            <p className="text-[var(--color-text-muted)] text-lg mb-4">No strategies found. Create one to get started!</p>
                        </div>
                    )}
                </div>
            )}
        </div>
    );
};

export default Dashboard;
