import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {
    Plus, Trash2, Calendar, TrendingUp, Download, Sparkles, BarChart3, Upload,
    Link as LinkIcon, CheckCircle, XCircle, Eye, Hash, Zap, Target, FileText,
    Settings, RefreshCw, Smartphone, MapPin, ChevronDown
} from 'lucide-react';
import {
    Chart as ChartJS,
    CategoryScale,
    LinearScale,
    BarElement,
    LineElement,
    PointElement,
    ArcElement,
    Title,
    Tooltip,
    Legend,
    Filler
} from 'chart.js';
import { Bar, Pie, Line } from 'react-chartjs-2';

// Register ChartJS components
ChartJS.register(
    CategoryScale,
    LinearScale,
    BarElement,
    LineElement,
    PointElement,
    ArcElement,
    Title,
    Tooltip,
    Legend,
    Filler
);

const PerformanceReports = () => {
    const [rows, setRows] = useState([
        { id: 1, url: '', publishDate: '', details: '' }
    ]);
    const [dateRange, setDateRange] = useState({
        startDate: '',
        endDate: ''
    });

    // Account connection states
    const [gscConnected, setGscConnected] = useState(false);
    const [ga4Connected, setGa4Connected] = useState(false);
    const [gscProperty, setGscProperty] = useState('');
    const [ga4Property, setGa4Property] = useState('');
    const [showPropertyModal, setShowPropertyModal] = useState(false);
    const [propertyPlatform, setPropertyPlatform] = useState('');

    // Data and loading states
    const [performanceData, setPerformanceData] = useState(null);
    const [loading, setLoading] = useState(false);
    const [aiInsights, setAiInsights] = useState('');
    const [recommendations, setRecommendations] = useState([]);
    const [generatingInsights, setGeneratingInsights] = useState(false);

    // Report templates
    const [showTemplatesModal, setShowTemplatesModal] = useState(false);
    const [templates, setTemplates] = useState([]);
    const [templateCategories, setTemplateCategories] = useState([]);
    const [selectedCategory, setSelectedCategory] = useState('all');

    // Chart data
    const [chartData, setChartData] = useState(null);
    const [selectedChartType, setSelectedChartType] = useState('bar_chart');

    // Current report ID (if viewing existing report)
    const [currentReportId, setCurrentReportId] = useState(null);

    // URL confirmation modal
    const [showConfirmationModal, setShowConfirmationModal] = useState(false);
    const [urlsToFetch, setUrlsToFetch] = useState([]);

    useEffect(() => {
        loadTemplates();
    }, []);

    const loadTemplates = async () => {
        try {
            const response = await axios.get('/api/report-templates');
            setTemplates(response.data.templates);
            setTemplateCategories(response.data.categories);
        } catch (err) {
            console.error('Error loading templates:', err);
        }
    };

    // Add a new row
    const addRow = () => {
        const newId = Math.max(...rows.map(r => r.id), 0) + 1;
        setRows([...rows, { id: newId, url: '', publishDate: '', details: '' }]);
    };

    // Remove a row
    const removeRow = (id) => {
        if (rows.length > 1) {
            setRows(rows.filter(row => row.id !== id));
        }
    };

    // Update row data
    const updateRow = (id, field, value) => {
        setRows(rows.map(row =>
            row.id === id ? { ...row, [field]: value } : row
        ));
    };

    // Connect GSC or GA4
    const connectAccount = async (platform) => {
        try {
            const response = await axios.post(`/api/performance-reports/${currentReportId}/connect-oauth`, {
                platform
            });

            // Redirect to OAuth flow
            window.location.href = response.data.authorization_url;
        } catch (err) {
            console.error('Error connecting account:', err);
            alert(`Error connecting ${platform}. Please try again.`);
        }
    };

    // Set property ID
    const setPropertyId = async () => {
        try {
            await axios.post(`/api/performance-reports/${currentReportId}/set-property`, {
                platform: propertyPlatform,
                property_id: propertyPlatform === 'GSC' ? gscProperty : ga4Property
            });

            setShowPropertyModal(false);

            if (propertyPlatform === 'GSC') {
                setGscConnected(true);
            } else {
                setGa4Connected(true);
            }
        } catch (err) {
            console.error('Error setting property:', err);
            alert('Error setting property ID');
        }
    };

    // Show confirmation modal before fetching
    const prepareDataFetch = () => {
        const validUrls = rows.filter(r => r.url).map(r => r.url);

        if (validUrls.length === 0) {
            alert('Please add at least one URL');
            return;
        }

        if (!dateRange.startDate || !dateRange.endDate) {
            alert('Please select date range');
            return;
        }

        if (!gscConnected && !ga4Connected) {
            alert('Please connect at least one account (GSC or GA4)');
            return;
        }

        setUrlsToFetch(validUrls);
        setShowConfirmationModal(true);
    };

    // Fetch performance data from GA4 and GSC
    const fetchPerformanceData = async () => {
        setShowConfirmationModal(false);
        setLoading(true);

        try {
            const urlsToFetch = rows.filter(r => r.url).map(r => r.url);

            // Create or update report
            let reportId = currentReportId;

            if (!reportId) {
                const createResponse = await axios.post('/api/performance-reports', {
                    name: `Performance Report - ${new Date().toLocaleDateString()}`,
                    urls: urlsToFetch,
                    start_date: dateRange.startDate,
                    end_date: dateRange.endDate
                });
                reportId = createResponse.data.id;
                setCurrentReportId(reportId);
            }

            // Prepare URL metadata
            const url_metadata = {};
            rows.forEach(row => {
                if (row.url) {
                    url_metadata[row.url] = {
                        publish_date: row.publishDate || null,
                        change_details: row.details || null
                    };
                }
            });

            // Fetch data
            const response = await axios.post(`/api/performance-reports/${reportId}/fetch-data`, {
                urls: urlsToFetch,
                start_date: dateRange.startDate,
                end_date: dateRange.endDate,
                url_metadata
            });

            setPerformanceData(response.data);
            setAiInsights(response.data.insights);
            setRecommendations(response.data.recommendations || []);

            // Load chart data
            loadChartData(reportId);
        } catch (err) {
            console.error('Error fetching performance data:', err);
            alert('Error fetching data: ' + (err.response?.data?.error || err.message));
        } finally {
            setLoading(false);
        }
    };

    // Load chart data
    const loadChartData = async (reportId) => {
        try {
            const response = await axios.get(`/api/performance-reports/${reportId}/chart-data`);
            setChartData(response.data);
        } catch (err) {
            console.error('Error loading chart data:', err);
        }
    };

    // Export data
    const exportData = async () => {
        if (!currentReportId) {
            alert('No report data to export');
            return;
        }

        window.open(`/api/performance-reports/${currentReportId}/export`, '_blank');
    };

    // Import from CSV/Excel
    const handleFileUpload = (e) => {
        const file = e.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (event) => {
            const content = event.target.result;
            const lines = content.split('\n');
            const newRows = [];

            // Skip header row
            for (let i = 1; i < lines.length; i++) {
                const line = lines[i].trim();
                if (!line) continue;

                const [date, details, ...rest] = line.split(',');

                // Try to find URL in the line
                let url = '';
                const urlMatch = line.match(/https?:\/\/[^\s,]+/);
                if (urlMatch) {
                    url = urlMatch[0];
                }

                if (date || details || url) {
                    newRows.push({
                        id: i,
                        url: url,
                        publishDate: date?.trim() || '',
                        details: details?.trim() || ''
                    });
                }
            }

            if (newRows.length > 0) {
                setRows(newRows);
            }
        };
        reader.readAsText(file);
    };

    // Generate report from template
    const generateFromTemplate = async (templateId) => {
        try {
            const validUrls = rows.filter(r => r.url).map(r => r.url);

            const response = await axios.post(`/api/report-templates/${templateId}/generate`, {
                urls: validUrls,
                start_date: dateRange.startDate,
                end_date: dateRange.endDate
            });

            setCurrentReportId(response.data.report_id);
            setShowTemplatesModal(false);
            alert(`Report created: ${response.data.message}`);
        } catch (err) {
            console.error('Error generating from template:', err);
            alert('Error creating report from template');
        }
    };

    const getChartIcon = (category) => {
        const icons = {
            performance: TrendingUp,
            technical: Settings,
            keywords: Hash,
            conversion: Target,
            competitive: Zap,
            rich_results: Sparkles,
            content: FileText,
            mobile: Smartphone,
            local: MapPin,
            executive: BarChart3
        };
        return icons[category] || FileText;
    };

    const filteredTemplates = selectedCategory === 'all'
        ? templates
        : templates.filter(t => t.category === selectedCategory);

    return (
        <div className="max-w-7xl mx-auto">
            <div className="mb-6 flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold text-white mb-2">Generate Performance Reports</h1>
                    <p className="text-[var(--color-text-muted)]">
                        Track URL performance with publish dates and change details from GA4 & GSC
                    </p>
                </div>
                <div className="flex gap-2">
                    <button
                        onClick={() => setShowTemplatesModal(true)}
                        className="px-4 py-2 bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 text-white rounded-lg flex items-center gap-2 transition-all shadow-lg"
                    >
                        <Sparkles size={18} />
                        Report Templates
                    </button>
                    <label className="px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] cursor-pointer flex items-center gap-2 transition-colors">
                        <Upload size={18} />
                        Import CSV
                        <input
                            type="file"
                            accept=".csv,.txt"
                            className="hidden"
                            onChange={handleFileUpload}
                        />
                    </label>
                    {performanceData && (
                        <button
                            onClick={exportData}
                            className="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg flex items-center gap-2 transition-colors"
                        >
                            <Download size={18} />
                            Export CSV
                        </button>
                    )}
                </div>
            </div>

            {/* Account Connection Section */}
            <div className="glass-card p-6 mb-6">
                <h2 className="text-xl font-bold text-white mb-4 flex items-center gap-2">
                    <LinkIcon size={20} />
                    Connect Accounts (Scoped to this Report)
                </h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {/* GSC Connection */}
                    <div className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-4">
                        <div className="flex items-center justify-between mb-3">
                            <h3 className="font-semibold text-white">Google Search Console</h3>
                            {gscConnected ? (
                                <CheckCircle size={20} className="text-green-400" />
                            ) : (
                                <XCircle size={20} className="text-gray-400" />
                            )}
                        </div>
                        {!gscConnected ? (
                            <button
                                onClick={() => connectAccount('GSC')}
                                className="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg flex items-center justify-center gap-2 transition-colors"
                            >
                                <LinkIcon size={16} />
                                Connect GSC
                            </button>
                        ) : (
                            <div>
                                <p className="text-sm text-green-400 mb-2">✓ Connected</p>
                                <button
                                    onClick={() => {
                                        setPropertyPlatform('GSC');
                                        setShowPropertyModal(true);
                                    }}
                                    className="w-full px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] transition-colors text-sm"
                                >
                                    {gscProperty ? `Property: ${gscProperty}` : 'Set Property URL'}
                                </button>
                            </div>
                        )}
                    </div>

                    {/* GA4 Connection */}
                    <div className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-4">
                        <div className="flex items-center justify-between mb-3">
                            <h3 className="font-semibold text-white">Google Analytics 4</h3>
                            {ga4Connected ? (
                                <CheckCircle size={20} className="text-green-400" />
                            ) : (
                                <XCircle size={20} className="text-gray-400" />
                            )}
                        </div>
                        {!ga4Connected ? (
                            <button
                                onClick={() => connectAccount('GA4')}
                                className="w-full px-4 py-2 bg-orange-600 hover:bg-orange-700 text-white rounded-lg flex items-center justify-center gap-2 transition-colors"
                            >
                                <LinkIcon size={16} />
                                Connect GA4
                            </button>
                        ) : (
                            <div>
                                <p className="text-sm text-green-400 mb-2">✓ Connected</p>
                                <button
                                    onClick={() => {
                                        setPropertyPlatform('GA4');
                                        setShowPropertyModal(true);
                                    }}
                                    className="w-full px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] transition-colors text-sm"
                                >
                                    {ga4Property ? `Property: ${ga4Property}` : 'Set Property ID'}
                                </button>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Date Range Selection */}
            <div className="glass-card p-6 mb-6">
                <h2 className="text-xl font-bold text-white mb-4 flex items-center gap-2">
                    <Calendar size={20} />
                    Data Collection Timeframe
                </h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 items-end">
                    <div>
                        <label className="block text-sm text-[var(--color-text-muted)] mb-2">Start Date</label>
                        <input
                            type="date"
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-2 text-white focus:border-[var(--color-primary)] outline-none"
                            value={dateRange.startDate}
                            onChange={e => setDateRange({ ...dateRange, startDate: e.target.value })}
                        />
                    </div>
                    <div>
                        <label className="block text-sm text-[var(--color-text-muted)] mb-2">End Date</label>
                        <input
                            type="date"
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-2 text-white focus:border-[var(--color-primary)] outline-none"
                            value={dateRange.endDate}
                            onChange={e => setDateRange({ ...dateRange, endDate: e.target.value })}
                        />
                    </div>
                    <button
                        onClick={prepareDataFetch}
                        disabled={loading || !dateRange.startDate || !dateRange.endDate}
                        className="px-6 py-2 bg-[var(--color-primary)] hover:bg-blue-600 disabled:bg-gray-600 disabled:cursor-not-allowed text-white rounded-lg flex items-center justify-center gap-2 transition-colors shadow-[var(--shadow-glow)]"
                    >
                        {loading ? (
                            <>Loading Data...</>
                        ) : (
                            <>
                                <TrendingUp size={18} />
                                Fetch Performance Data
                            </>
                        )}
                    </button>
                </div>
            </div>

            {/* URL Tracking Table */}
            <div className="glass-card p-6 mb-6">
                <div className="flex items-center justify-between mb-4">
                    <h2 className="text-xl font-bold text-white">URLs & Changes</h2>
                    <button
                        onClick={addRow}
                        className="px-4 py-2 bg-[var(--color-primary)] hover:bg-blue-600 text-white rounded-lg flex items-center gap-2 transition-colors"
                    >
                        <Plus size={18} />
                        Add Row
                    </button>
                </div>

                <div className="overflow-x-auto">
                    <table className="w-full">
                        <thead>
                            <tr className="border-b border-[var(--color-border)]">
                                <th className="text-left text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2">URL</th>
                                <th className="text-left text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2 w-40">Publish Date</th>
                                <th className="text-left text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2">Details of Change</th>
                                {performanceData && (
                                    <>
                                        <th className="text-right text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2 w-24">Clicks</th>
                                        <th className="text-right text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2 w-24">Impr.</th>
                                        <th className="text-right text-sm font-medium text-[var(--color-text-muted)] pb-3 px-2 w-24">Sessions</th>
                                    </>
                                )}
                                <th className="w-12"></th>
                            </tr>
                        </thead>
                        <tbody>
                            {rows.map((row) => {
                                return (
                                    <tr key={row.id} className="border-b border-[var(--color-border)] hover:bg-[rgba(255,255,255,0.02)]">
                                        <td className="py-3 px-2">
                                            <input
                                                type="url"
                                                placeholder="https://example.com/page"
                                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white text-sm focus:border-[var(--color-primary)] outline-none"
                                                value={row.url}
                                                onChange={e => updateRow(row.id, 'url', e.target.value)}
                                            />
                                        </td>
                                        <td className="py-3 px-2">
                                            <input
                                                type="date"
                                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white text-sm focus:border-[var(--color-primary)] outline-none"
                                                value={row.publishDate}
                                                onChange={e => updateRow(row.id, 'publishDate', e.target.value)}
                                            />
                                        </td>
                                        <td className="py-3 px-2">
                                            <input
                                                type="text"
                                                placeholder="What changed..."
                                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white text-sm focus:border-[var(--color-primary)] outline-none"
                                                value={row.details}
                                                onChange={e => updateRow(row.id, 'details', e.target.value)}
                                            />
                                        </td>
                                        {performanceData && (
                                            <>
                                                <td className="py-3 px-2 text-right text-white font-medium">-</td>
                                                <td className="py-3 px-2 text-right text-white font-medium">-</td>
                                                <td className="py-3 px-2 text-right text-white font-medium">-</td>
                                            </>
                                        )}
                                        <td className="py-3 px-2">
                                            <button
                                                onClick={() => removeRow(row.id)}
                                                className="p-2 text-red-400 hover:text-red-300 hover:bg-[rgba(255,0,0,0.1)] rounded transition-colors"
                                                disabled={rows.length === 1}
                                            >
                                                <Trash2 size={16} />
                                            </button>
                                        </td>
                                    </tr>
                                );
                            })}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Performance Visualization */}
            {performanceData && chartData && (
                <div className="glass-card p-6 mb-6">
                    <div className="flex items-center justify-between mb-4">
                        <h2 className="text-xl font-bold text-white flex items-center gap-2">
                            <BarChart3 size={20} />
                            Performance Charts
                        </h2>
                        <select
                            value={selectedChartType}
                            onChange={(e) => setSelectedChartType(e.target.value)}
                            className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-2 text-white focus:border-[var(--color-primary)] outline-none"
                        >
                            <option value="bar_chart">Bar Chart - Traffic</option>
                            <option value="ctr_performance">CTR Performance</option>
                            <option value="position_ranking">Position Rankings</option>
                            <option value="pie_chart">Pie Chart - Distribution</option>
                            <option value="cwv_metrics">Core Web Vitals</option>
                        </select>
                    </div>

                    <div className="bg-[rgba(0,0,0,0.2)] border border-[var(--color-border)] rounded-lg p-6">
                        {selectedChartType === 'bar_chart' && (
                            <Bar data={chartData.bar_chart} options={{
                                responsive: true,
                                plugins: {
                                    legend: { display: true, labels: { color: '#fff' } },
                                    title: { display: true, text: 'Clicks vs Sessions', color: '#fff' }
                                },
                                scales: {
                                    x: { ticks: { color: '#9ca3af' } },
                                    y: { ticks: { color: '#9ca3af' } }
                                }
                            }} />
                        )}
                        {selectedChartType === 'ctr_performance' && (
                            <Line data={chartData.ctr_performance} options={{
                                responsive: true,
                                plugins: {
                                    legend: { display: true, labels: { color: '#fff' } },
                                    title: { display: true, text: 'CTR Performance', color: '#fff' }
                                },
                                scales: {
                                    x: { ticks: { color: '#9ca3af' } },
                                    y: { ticks: { color: '#9ca3af' } }
                                }
                            }} />
                        )}
                        {selectedChartType === 'position_ranking' && (
                            <Bar data={chartData.position_ranking} options={{
                                responsive: true,
                                plugins: {
                                    legend: { display: true, labels: { color: '#fff' } },
                                    title: { display: true, text: 'Average Position', color: '#fff' }
                                },
                                scales: {
                                    x: { ticks: { color: '#9ca3af' } },
                                    y: { reverse: true, ticks: { color: '#9ca3af' } }
                                }
                            }} />
                        )}
                        {selectedChartType === 'pie_chart' && (
                            <div className="max-w-md mx-auto">
                                <Pie data={chartData.pie_chart} options={{
                                    responsive: true,
                                    plugins: {
                                        legend: { display: true, labels: { color: '#fff' }, position: 'bottom' },
                                        title: { display: true, text: 'Traffic Distribution', color: '#fff' }
                                    }
                                }} />
                            </div>
                        )}
                        {selectedChartType === 'cwv_metrics' && (
                            <Bar data={chartData.cwv_metrics} options={{
                                responsive: true,
                                plugins: {
                                    legend: { display: true, labels: { color: '#fff' } },
                                    title: { display: true, text: 'Core Web Vitals Averages', color: '#fff' }
                                },
                                scales: {
                                    x: { ticks: { color: '#9ca3af' } },
                                    y: { ticks: { color: '#9ca3af' } }
                                }
                            }} />
                        )}
                    </div>

                    {/* Totals */}
                    <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mt-6">
                        <div className="bg-[rgba(59,130,246,0.1)] border border-blue-500/30 rounded-lg p-4">
                            <div className="text-sm text-blue-400 mb-1">Total Clicks</div>
                            <div className="text-2xl font-bold text-white">
                                {performanceData.totals?.clicks?.toLocaleString() || 0}
                            </div>
                        </div>
                        <div className="bg-[rgba(16,185,129,0.1)] border border-emerald-500/30 rounded-lg p-4">
                            <div className="text-sm text-emerald-400 mb-1">Total Impressions</div>
                            <div className="text-2xl font-bold text-white">
                                {performanceData.totals?.impressions?.toLocaleString() || 0}
                            </div>
                        </div>
                        <div className="bg-[rgba(251,146,60,0.1)] border border-orange-500/30 rounded-lg p-4">
                            <div className="text-sm text-orange-400 mb-1">Total Sessions</div>
                            <div className="text-2xl font-bold text-white">
                                {performanceData.totals?.sessions?.toLocaleString() || 0}
                            </div>
                        </div>
                        <div className="bg-[rgba(168,85,247,0.1)] border border-purple-500/30 rounded-lg p-4">
                            <div className="text-sm text-purple-400 mb-1">Avg CTR</div>
                            <div className="text-2xl font-bold text-white">
                                {performanceData.totals?.ctr?.toFixed(2) || 0}%
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* AI Insights Section */}
            {aiInsights && (
                <div className="glass-card p-6 mb-6">
                    <h2 className="text-xl font-bold text-white mb-4 flex items-center gap-2">
                        <Sparkles size={20} />
                        AI-Generated Insights
                    </h2>
                    <div className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-4">
                        <div className="text-white whitespace-pre-wrap prose prose-invert max-w-none">
                            {aiInsights}
                        </div>
                    </div>
                </div>
            )}

            {/* Recommendations Section */}
            {recommendations && recommendations.length > 0 && (
                <div className="glass-card p-6 mb-6">
                    <h2 className="text-xl font-bold text-white mb-4 flex items-center gap-2">
                        <Target size={20} />
                        Actionable Recommendations
                    </h2>
                    <div className="space-y-4">
                        {recommendations.map((rec, idx) => (
                            <div
                                key={idx}
                                className={`border-l-4 p-4 rounded-r-lg ${rec.type === 'critical' ? 'border-red-500 bg-[rgba(239,68,68,0.1)]' :
                                        rec.type === 'important' ? 'border-orange-500 bg-[rgba(251,146,60,0.1)]' :
                                            'border-blue-500 bg-[rgba(59,130,246,0.1)]'
                                    }`}
                            >
                                <div className="flex items-start justify-between mb-2">
                                    <h3 className="font-bold text-white">{rec.title}</h3>
                                    <span className={`text-xs px-2 py-1 rounded ${rec.type === 'critical' ? 'bg-red-500/20 text-red-300' :
                                            rec.type === 'important' ? 'bg-orange-500/20 text-orange-300' :
                                                'bg-blue-500/20 text-blue-300'
                                        }`}>
                                        {rec.category}
                                    </span>
                                </div>
                                <p className="text-sm text-[var(--color-text-muted)] mb-3">{rec.description}</p>
                                <div className="space-y-1">
                                    {rec.actions && rec.actions.map((action, aidx) => (
                                        <div key={aidx} className="flex items-start gap-2 text-sm text-white">
                                            <span className="text-green-400 mt-0.5">✓</span>
                                            <span>{action}</span>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            )}

            {/* Report Templates Modal */}
            {showTemplatesModal && (
                <div className="fixed inset-0 bg-black/70 flex items-center justify-center z-50 p-4">
                    <div className="bg-[var(--color-surface)] border border-[var(--color-border)] rounded-xl max-w-4xl w-full max-h-[90vh] overflow-hidden flex flex-col">
                        <div className="p-6 border-b border-[var(--color-border)]">
                            <h2 className="text-2xl font-bold text-white mb-2">Report Templates</h2>
                            <p className="text-[var(--color-text-muted)]">Choose a pre-configured template for your analysis</p>

                            {/* Category Filter */}
                            <div className="flex gap-2 mt-4 overflow-x-auto pb-2">
                                <button
                                    onClick={() => setSelectedCategory('all')}
                                    className={`px-4 py-2 rounded-lg whitespace-nowrap transition-colors ${selectedCategory === 'all'
                                            ? 'bg-[var(--color-primary)] text-white'
                                            : 'bg-[rgba(255,255,255,0.05)] text-[var(--color-text-muted)] hover:bg-[rgba(255,255,255,0.1)]'
                                        }`}
                                >
                                    All Templates
                                </button>
                                {templateCategories.map(cat => (
                                    <button
                                        key={cat.key}
                                        onClick={() => setSelectedCategory(cat.key)}
                                        className={`px-4 py-2 rounded-lg whitespace-nowrap transition-colors ${selectedCategory === cat.key
                                                ? 'bg-[var(--color-primary)] text-white'
                                                : 'bg-[rgba(255,255,255,0.05)] text-[var(--color-text-muted)] hover:bg-[rgba(255,255,255,0.1)]'
                                            }`}
                                    >
                                        {cat.label} ({cat.count})
                                    </button>
                                ))}
                            </div>
                        </div>

                        <div className="p-6 overflow-y-auto flex-1">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {filteredTemplates.map(template => {
                                    const Icon = getChartIcon(template.category);
                                    return (
                                        <div
                                            key={template.id}
                                            className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-4 hover:border-[var(--color-primary)] transition-colors cursor-pointer"
                                            onClick={() => generateFromTemplate(template.id)}
                                        >
                                            <div className="flex items-start gap-3 mb-3">
                                                <div className="p-2 bg-[var(--color-primary)]/20 rounded-lg">
                                                    <Icon size={20} className="text-[var(--color-primary)]" />
                                                </div>
                                                <div className="flex-1">
                                                    <h3 className="font-bold text-white mb-1">{template.name}</h3>
                                                    <p className="text-sm text-[var(--color-text-muted)]">{template.description}</p>
                                                </div>
                                            </div>

                                            <div className="text-xs text-[var(--color-text-muted)] space-y-2">
                                                <div>
                                                    <span className="font-semibold text-white">Insights:</span>
                                                    <ul className="mt-1 space-y-0.5 ml-4">
                                                        {template.insights_generated?.slice(0, 3).map((insight, idx) => (
                                                            <li key={idx} className="list-disc">{insight}</li>
                                                        ))}
                                                    </ul>
                                                </div>
                                            </div>
                                        </div>
                                    );
                                })}
                            </div>
                        </div>

                        <div className="p-6 border-t border-[var(--color-border)]">
                            <button
                                onClick={() => setShowTemplatesModal(false)}
                                className="w-full px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] transition-colors"
                            >
                                Close
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Property ID Modal */}
            {showPropertyModal && (
                <div className="fixed inset-0 bg-black/70 flex items-center justify-center z-50 p-4">
                    <div className="bg-[var(--color-surface)] border border-[var(--color-border)] rounded-xl max-w-md w-full p-6">
                        <h2 className="text-xl font-bold text-white mb-4">
                            Set {propertyPlatform} Property
                        </h2>
                        <div className="mb-4">
                            <label className="block text-sm text-[var(--color-text-muted)] mb-2">
                                {propertyPlatform === 'GSC' ? 'Site URL (e.g., sc-domain:example.com)' : 'Property ID (e.g., 123456789)'}
                            </label>
                            <input
                                type="text"
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-2 text-white focus:border-[var(--color-primary)] outline-none"
                                value={propertyPlatform === 'GSC' ? gscProperty : ga4Property}
                                onChange={e => propertyPlatform === 'GSC' ? setGscProperty(e.target.value) : setGa4Property(e.target.value)}
                                placeholder={propertyPlatform === 'GSC' ? 'sc-domain:example.com' : '123456789'}
                            />
                        </div>
                        <div className="flex gap-3">
                            <button
                                onClick={setPropertyId}
                                className="flex-1 px-4 py-2 bg-[var(--color-primary)] hover:bg-blue-600 text-white rounded-lg transition-colors"
                            >
                                Save
                            </button>
                            <button
                                onClick={() => setShowPropertyModal(false)}
                                className="flex-1 px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] transition-colors"
                            >
                                Cancel
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Confirmation Modal */}
            {showConfirmationModal && (
                <div className="fixed inset-0 bg-black/70 flex items-center justify-center z-50 p-4">
                    <div className="bg-[var(--color-surface)] border border-[var(--color-border)] rounded-xl max-w-lg w-full p-6">
                        <h2 className="text-xl font-bold text-white mb-4">Confirm Data Fetch</h2>
                        <div className="mb-4 space-y-3">
                            <p className="text-white">
                                You are about to fetch performance data for <span className="font-bold text-[var(--color-primary)]">{urlsToFetch.length} URL(s)</span>
                            </p>
                            <div className="bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-3 max-h-40 overflow-y-auto">
                                {urlsToFetch.map((url, idx) => (
                                    <div key={idx} className="text-sm text-[var(--color-text-muted)] truncate">
                                        • {url}
                                    </div>
                                ))}
                            </div>
                            <p className="text-sm text-[var(--color-text-muted)]">
                                Date range: <span className="text-white">{dateRange.startDate} to {dateRange.endDate}</span>
                            </p>
                            <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-lg p-3">
                                <p className="text-sm text-yellow-300">
                                    ⚠️ This will fetch comprehensive data including clicks, impressions, CWV, queries, and rich results.
                                </p>
                            </div>
                        </div>
                        <div className="flex gap-3">
                            <button
                                onClick={fetchPerformanceData}
                                className="flex-1 px-4 py-2 bg-[var(--color-primary)] hover:bg-blue-600 text-white rounded-lg transition-colors font-semibold"
                            >
                                Confirm & Fetch
                            </button>
                            <button
                                onClick={() => setShowConfirmationModal(false)}
                                className="flex-1 px-4 py-2 bg-[var(--color-surface)] border border-[var(--color-border)] text-white rounded-lg hover:bg-[rgba(255,255,255,0.1)] transition-colors"
                            >
                                Cancel
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

export default PerformanceReports;
