import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { PieChart, Pie, Cell, ResponsiveContainer, BarChart, Bar, XAxis, YAxis, Tooltip, Legend } from 'recharts';
import { Save, Server, FileText, Palette, Check } from 'lucide-react';

const THEMES = [
    { name: 'Midnight Blue', primary: '#3b82f6', secondary: '#8b5cf6', bg: '#0f172a', text: '#f8fafc' },
    { name: 'Emerald City', primary: '#10b981', secondary: '#34d399', bg: '#022c22', text: '#ecfdf5' },
    { name: 'Sunset Glow', primary: '#f97316', secondary: '#f43f5e', bg: '#450a0a', text: '#fff1f2' },
    { name: 'Cyberpunk', primary: '#d946ef', secondary: '#8b5cf6', bg: '#2e1065', text: '#fae8ff' },
    { name: 'Slate Minimal', primary: '#94a3b8', secondary: '#cbd5e1', bg: '#1e293b', text: '#f1f5f9' },
    { name: 'Golden Hour', primary: '#eab308', secondary: '#f59e0b', bg: '#422006', text: '#fefce8' },
    { name: 'Royal Purple', primary: '#a855f7', secondary: '#d8b4fe', bg: '#3b0764', text: '#f3e8ff' },
    { name: 'Ocean Breeze', primary: '#06b6d4', secondary: '#22d3ee', bg: '#083344', text: '#ecfeff' },
    { name: 'Rose Garden', primary: '#ec4899', secondary: '#fbcfe8', bg: '#831843', text: '#fdf2f8' },
    { name: 'Deep Space', primary: '#6366f1', secondary: '#818cf8', bg: '#1e1b4b', text: '#e0e7ff' }
];

const Settings = () => {
    const [activeTab, setActiveTab] = useState('look-feel');
    const [systemStats, setSystemStats] = useState(null);
    const [loadingStats, setLoadingStats] = useState(false);
    const [toast, setToast] = useState(null);

    // Inject Bootstrap 5
    useEffect(() => {
        const link = document.createElement('link');
        link.href = "https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css";
        link.rel = "stylesheet";
        link.id = "bootstrap-css";
        document.head.appendChild(link);

        const script = document.createElement('script');
        script.src = "https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js";
        script.id = "bootstrap-js";
        document.body.appendChild(script);

        return () => {
            const existingCss = document.getElementById("bootstrap-css");
            if (existingCss) existingCss.remove();

            const existingJs = document.getElementById("bootstrap-js");
            if (existingJs) existingJs.remove();
        };
    }, []);

    // Fetch Stats
    useEffect(() => {
        if (activeTab === 'data') {
            fetchStats();
        }
    }, [activeTab]);

    const fetchStats = async () => {
        setLoadingStats(true);
        try {
            const res = await axios.get('/api/system/stats');
            setSystemStats(res.data);
        } catch (err) {
            console.error("Failed to fetch stats", err);
        } finally {
            setLoadingStats(false);
        }
    };

    const applyTheme = (theme) => {
        const root = document.documentElement;
        root.style.setProperty('--color-primary', theme.primary);
        root.style.setProperty('--color-secondary', theme.secondary);
        root.style.setProperty('--color-bg', theme.bg);

        // Save
        localStorage.setItem('user-theme', JSON.stringify(theme));

        // Toast
        setToast(`Theme "${theme.name}" applied and saved!`);
        setTimeout(() => setToast(null), 3000);
    };

    const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042'];

    return (
        <div className="container-fluid p-4 text-white">
            <h2 className="mb-4">Settings</h2>

            {toast && (
                <div className="toast-container position-fixed bottom-0 end-0 p-3">
                    <div className="toast show align-items-center text-white bg-success border-0">
                        <div className="d-flex">
                            <div className="toast-body">{toast}</div>
                            <button type="button" className="btn-close btn-close-white me-2 m-auto" onClick={() => setToast(null)}></button>
                        </div>
                    </div>
                </div>
            )}

            <div className="card bg-dark text-white shadow-sm" style={{ border: '1px solid rgba(255,255,255,0.1)' }}>
                <div className="card-header">
                    <ul className="nav nav-tabs card-header-tabs">
                        <li className="nav-item">
                            <button
                                className={`nav-link ${activeTab === 'look-feel' ? 'active bg-secondary text-white' : 'text-muted'}`}
                                onClick={() => setActiveTab('look-feel')}
                            >
                                <Palette size={18} className="d-inline-block me-2" /> Look & Feel
                            </button>
                        </li>
                        <li className="nav-item">
                            <button
                                className={`nav-link ${activeTab === 'data' ? 'active bg-secondary text-white' : 'text-muted'}`}
                                onClick={() => setActiveTab('data')}
                            >
                                <Server size={18} className="d-inline-block me-2" /> Data & Storage
                            </button>
                        </li>
                        <li className="nav-item">
                            <button
                                className={`nav-link ${activeTab === 'notes' ? 'active bg-secondary text-white' : 'text-muted'}`}
                                onClick={() => setActiveTab('notes')}
                            >
                                <FileText size={18} className="d-inline-block me-2" /> Admin Notes
                            </button>
                        </li>
                    </ul>
                </div>

                <div className="card-body p-4">
                    {activeTab === 'look-feel' && (
                        <div>
                            <h4>Theme Customizer</h4>
                            <p className="text-secondary mb-4">Select a theme to preview in real-time. Changes are auto-saved.</p>
                            <div className="row g-4">
                                {THEMES.map((theme, idx) => (
                                    <div className="col-md-4 col-lg-3" key={idx}>
                                        <div
                                            className="card h-100 cursor-pointer theme-card border-0"
                                            style={{ backgroundColor: theme.bg, border: `2px solid ${theme.primary}` }}
                                            onClick={() => applyTheme(theme)}
                                        >
                                            <div className="card-body text-center">
                                                <div className="mb-3 d-flex justify-content-center gap-2">
                                                    <div style={{ width: 20, height: 20, borderRadius: '50%', background: theme.primary }}></div>
                                                    <div style={{ width: 20, height: 20, borderRadius: '50%', background: theme.secondary }}></div>
                                                </div>
                                                <h5 style={{ color: theme.text }}>{theme.name}</h5>
                                            </div>
                                            <div className="card-footer border-top-0 d-grid">
                                                <button className="btn btn-sm btn-outline-light">Apply</button>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}

                    {activeTab === 'data' && (
                        <div>
                            <h4>System Metrics</h4>
                            {!systemStats ? (
                                <p>Loading data...</p>
                            ) : (
                                <div className="row g-4 mt-2">
                                    <div className="col-md-6">
                                        <div className="card bg-secondary text-white h-100">
                                            <div className="card-body">
                                                <h5 className="card-title">Storage Usage</h5>
                                                <div style={{ height: 250 }}>
                                                    <ResponsiveContainer width="100%" height="100%">
                                                        <PieChart>
                                                            <Pie
                                                                data={[
                                                                    { name: 'Used', value: systemStats.storage.used },
                                                                    { name: 'Free', value: systemStats.storage.free }
                                                                ]}
                                                                cx="50%"
                                                                cy="50%"
                                                                innerRadius={60}
                                                                outerRadius={80}
                                                                fill="#8884d8"
                                                                paddingAngle={5}
                                                                dataKey="value"
                                                            >
                                                                {[{ name: 'Used', value: systemStats.storage.used }, { name: 'Free', value: systemStats.storage.free }].map((entry, index) => (
                                                                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                                                                ))}
                                                            </Pie>
                                                            <Tooltip />
                                                            <Legend />
                                                        </PieChart>
                                                    </ResponsiveContainer>
                                                </div>
                                                <p className="text-center mt-2">
                                                    {systemStats.storage.percent}% Used
                                                    <br />
                                                    <small className="text-light">
                                                        Total: {(systemStats.storage.total / 1e9).toFixed(2)} GB
                                                    </small>
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="col-md-6">
                                        <div className="card bg-secondary text-white h-100">
                                            <div className="card-body">
                                                <h5 className="card-title">Activity Metrics</h5>
                                                <div style={{ height: 250 }}>
                                                    <ResponsiveContainer width="100%" height="100%">
                                                        <BarChart
                                                            data={[
                                                                { name: 'Strategies', count: systemStats.metrics.strategies },
                                                                { name: 'Integrations', count: systemStats.metrics.integrations },
                                                                { name: 'Tests Tracked', count: systemStats.metrics.tests_tracked },
                                                            ]}
                                                            margin={{ top: 20, right: 30, left: 20, bottom: 5 }}
                                                        >
                                                            <XAxis dataKey="name" stroke="#fff" />
                                                            <YAxis stroke="#fff" />
                                                            <Tooltip cursor={{ fill: 'rgba(255,255,255,0.1)' }} contentStyle={{ backgroundColor: '#333', color: '#fff' }} />
                                                            <Bar dataKey="count" fill="#82ca9d" />
                                                        </BarChart>
                                                    </ResponsiveContainer>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            )}
                        </div>
                    )}

                    {activeTab === 'notes' && (
                        <div>
                            <h4>Admin Documentation</h4>
                            <div className="accordion mt-4 text-dark" id="adminAccordion">
                                <div className="accordion-item">
                                    <h2 className="accordion-header">
                                        <button className="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#collapseOne">
                                            System Overview
                                        </button>
                                    </h2>
                                    <div id="collapseOne" className="accordion-collapse collapse show" data-bs-parent="#adminAccordion">
                                        <div className="accordion-body">
                                            <strong>SEO Tracker V2</strong> is a comprehensive reporting tool. It connects to GSC and GA4 to provide unified insights.
                                            <br />
                                            The system is built with React (Frontend) and Flask (Backend).
                                        </div>
                                    </div>
                                </div>
                                <div className="accordion-item">
                                    <h2 className="accordion-header">
                                        <button className="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseTwo">
                                            Key Functionalities
                                        </button>
                                    </h2>
                                    <div id="collapseTwo" className="accordion-collapse collapse" data-bs-parent="#adminAccordion">
                                        <div className="accordion-body">
                                            <ul>
                                                <li><strong>Strategies:</strong> Define and track SEO campaigns.</li>
                                                <li><strong>Integrations:</strong> Connect Google accounts via OAuth.</li>
                                                <li><strong>Reporting:</strong> Generate PowerPoint slides and view real-time dashboards.</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                                <div className="accordion-item">
                                    <h2 className="accordion-header">
                                        <button className="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseThree">
                                            Data Retention & Privacy
                                        </button>
                                    </h2>
                                    <div id="collapseThree" className="accordion-collapse collapse" data-bs-parent="#adminAccordion">
                                        <div className="accordion-body">
                                            All data is stored locally on the server. OAuth tokens are encrypted. Backups occur daily at midnight.
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>

        </div>
    );
};

export default Settings;
