import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import axios from 'axios';
import { ArrowLeft, CheckCircle, XCircle, RefreshCw, BarChart2, FileText, Pencil, Save, X } from 'lucide-react';
import { motion } from 'framer-motion';
import Markdown from 'react-markdown';
import { ChevronDown, Plus } from 'lucide-react';
import DataImportPanel from '../components/DataImportPanel';

const StrategyDetail = () => {
    const { id } = useParams();
    const [strategy, setStrategy] = useState(null);
    const [loading, setLoading] = useState(true);
    const [isEditing, setIsEditing] = useState(false);
    const [editForm, setEditForm] = useState(null);
    const [showIntegrationsMenu, setShowIntegrationsMenu] = useState(false);
    const [newNote, setNewNote] = useState('');
    const [addingNote, setAddingNote] = useState(false);

    useEffect(() => {
        fetchStrategy();
    }, [id]);

    const fetchStrategy = async () => {
        try {
            const res = await axios.get(`/api/strategies/${id}`);
            setStrategy(res.data);
        } catch (err) {
            console.error("Failed to fetch strategy", err);
        } finally {
            setLoading(false);
        }
    };

    const handleConnect = (platform) => {
        // Initiate OAuth flow
        // Use relative path for production compatibility
        window.location.href = `/v2/api/auth/google/login/${id}?platform=${platform}`;
    };

    if (loading) return <div className="text-center py-20 text-[var(--color-text-muted)]">Loading strategy details...</div>;
    if (!strategy) return <div className="text-center py-20 text-red-400">Strategy not found</div>;

    const getIntegrationStatus = (platform) => {
        const integration = strategy.integrations.find(i => i.platform === platform);
        return integration && integration.connected;
    };

    const handleGenerateSlides = async () => {
        try {
            const response = await axios.post(`/api/strategies/${id}/generate-slides`, {}, { responseType: 'blob' });
            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', `strategy_${id}.pptx`);
            document.body.appendChild(link);
            link.click();
        } catch (err) {
            console.error("Failed to generate slides", err);
            alert("Failed to generate slides");
        }
    };

    const handleEditClick = () => {
        setEditForm({
            name: strategy.name,
            description: strategy.description,
            update_frequency: strategy.update_frequency,
            monitored_urls: Array.isArray(strategy.monitored_urls) ? strategy.monitored_urls.join('\n') : '',
            kpis: strategy.kpis || { clicks: '', impressions: '', conversions: '' }
        });
        setIsEditing(true);
    };

    const handleSaveEdit = async () => {
        try {
            await axios.put(`/api/strategies/${id}`, editForm);
            // Re-fetch to get the new timeline log
            fetchStrategy();
            setIsEditing(false);
        } catch (err) {
            console.error("Failed to update strategy", err);
            alert("Failed to update strategy");
        }
    };

    const handleSaveNote = async () => {
        if (!newNote.trim()) return;
        setAddingNote(true);
        try {
            const res = await axios.post(`/api/strategies/${id}/updates`, {
                description: newNote,
                impact_score: 0
            });
            setStrategy({
                ...strategy,
                updates: [res.data.update, ...(strategy.updates || [])]
            });
            setNewNote('');
        } catch (err) {
            console.error("Failed to save note", err);
            alert("Failed to save note");
        } finally {
            setAddingNote(false);
        }
    };

    const EditModal = () => {
        if (!isEditing) return null;

        return (
            <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
                <div className="glass-card w-full max-w-2xl max-h-[90vh] overflow-y-auto p-6 relative bg-[#1e293b]">
                    <button
                        onClick={() => setIsEditing(false)}
                        className="absolute top-4 right-4 text-gray-400 hover:text-white"
                    >
                        <X size={24} />
                    </button>

                    <h2 className="text-2xl font-bold mb-6">Edit Strategy</h2>

                    <div className="space-y-4">
                        <div>
                            <label className="block text-sm font-medium mb-1">Strategy Name</label>
                            <input
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none focus:border-[var(--color-primary)]"
                                value={editForm.name}
                                onChange={e => setEditForm({ ...editForm, name: e.target.value })}
                            />
                        </div>

                        <div>
                            <label className="block text-sm font-medium mb-1">Description</label>
                            <textarea
                                rows="3"
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none focus:border-[var(--color-primary)]"
                                value={editForm.description}
                                onChange={e => setEditForm({ ...editForm, description: e.target.value })}
                            />
                        </div>

                        <div>
                            <label className="block text-sm font-medium mb-1">Update Frequency</label>
                            <select
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none focus:border-[var(--color-primary)]"
                                value={editForm.update_frequency}
                                onChange={e => setEditForm({ ...editForm, update_frequency: e.target.value })}
                            >
                                <option value="daily">Daily</option>
                                <option value="weekly">Weekly</option>
                                <option value="monthly">Monthly</option>
                            </select>
                        </div>

                        <div>
                            <label className="block text-sm font-medium mb-1">Monitored URLs (one per line)</label>
                            <textarea
                                rows="4"
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none focus:border-[var(--color-primary)] font-mono text-sm"
                                value={editForm.monitored_urls}
                                onChange={e => setEditForm({ ...editForm, monitored_urls: e.target.value })}
                            />
                        </div>

                        <div className="p-3 border border-[var(--color-border)] rounded bg-[rgba(0,0,0,0.1)]">
                            <label className="block text-sm font-bold mb-2">Target KPIs (Monthly)</label>
                            <div className="grid grid-cols-3 gap-3">
                                <div>
                                    <label className="block text-xs text-gray-400 mb-1">Clicks</label>
                                    <input
                                        type="number"
                                        className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-2 py-1 text-white outline-none"
                                        value={editForm.kpis?.clicks || ''}
                                        onChange={e => setEditForm({ ...editForm, kpis: { ...editForm.kpis, clicks: e.target.value } })}
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs text-gray-400 mb-1">Impressions</label>
                                    <input
                                        type="number"
                                        className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-2 py-1 text-white outline-none"
                                        value={editForm.kpis?.impressions || ''}
                                        onChange={e => setEditForm({ ...editForm, kpis: { ...editForm.kpis, impressions: e.target.value } })}
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs text-gray-400 mb-1">Conversions</label>
                                    <input
                                        type="number"
                                        className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-2 py-1 text-white outline-none"
                                        value={editForm.kpis?.conversions || ''}
                                        onChange={e => setEditForm({ ...editForm, kpis: { ...editForm.kpis, conversions: e.target.value } })}
                                    />
                                </div>
                            </div>
                        </div>

                        <div className="flex justify-end gap-3 pt-2">
                            <button
                                onClick={() => setIsEditing(false)}
                                className="px-4 py-2 hover:bg-white/10 rounded transition-colors"
                            >
                                Cancel
                            </button>
                            <button
                                onClick={handleSaveEdit}
                                className="px-4 py-2 bg-[var(--color-primary)] hover:bg-blue-600 text-white rounded flex items-center gap-2 transition-colors"
                            >
                                <Save size={18} /> Save Changes
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        );
    };

    return (
        <div className="space-y-6">
            <Link to="/" className="flex items-center gap-2 text-[var(--color-text-muted)] hover:text-white transition-colors">
                <ArrowLeft size={20} /> Back to Dashboard
            </Link>

            {/* Header */}
            <div className="glass-card p-6 flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <div>
                    <h1 className="text-3xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-white to-gray-400">{strategy.name}</h1>
                    <p className="text-[var(--color-text-muted)] mt-1">{strategy.description}</p>
                </div>
                <div className="flex gap-2 relative">
                    {/* Integrations Dropdown */}
                    <div className="relative">
                        <button
                            onClick={() => setShowIntegrationsMenu(!showIntegrationsMenu)}
                            className="flex items-center gap-2 bg-[rgba(255,255,255,0.1)] hover:bg-[rgba(255,255,255,0.2)] px-4 py-2 rounded-[var(--radius-md)] transition-colors border border-transparent hover:border-[var(--color-border)]"
                        >
                            <RefreshCw size={18} /> Integrations <ChevronDown size={14} />
                        </button>

                        {showIntegrationsMenu && (
                            <div className="absolute top-full right-0 mt-2 w-72 bg-[#1e293b] border border-[var(--color-border)] rounded-[var(--radius-md)] shadow-xl z-50 p-4 space-y-4">
                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <h4 className="font-bold text-sm">Search Console</h4>
                                        {getIntegrationStatus('GSC') ? <CheckCircle size={16} className="text-green-400" /> : <XCircle size={16} className="text-gray-500" />}
                                    </div>
                                    {!getIntegrationStatus('GSC') && (
                                        <button onClick={() => handleConnect('GSC')} className="w-full py-1.5 text-xs bg-orange-500/20 text-orange-400 hover:bg-orange-500/30 rounded transition-colors">Connect GSC</button>
                                    )}
                                </div>
                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <h4 className="font-bold text-sm">Google Analytics 4</h4>
                                        {getIntegrationStatus('GA4') ? <CheckCircle size={16} className="text-green-400" /> : <XCircle size={16} className="text-gray-500" />}
                                    </div>
                                    {!getIntegrationStatus('GA4') && (
                                        <button onClick={() => handleConnect('GA4')} className="w-full py-1.5 text-xs bg-yellow-500/20 text-yellow-400 hover:bg-yellow-500/30 rounded transition-colors">Connect GA4</button>
                                    )}
                                </div>
                            </div>
                        )}
                    </div>

                    <button
                        onClick={handleEditClick}
                        className="flex items-center gap-2 bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded-[var(--radius-md)] transition-colors shadow-lg text-sm"
                    >
                        <Pencil size={16} /> <span>Edit</span>
                    </button>

                    <button
                        onClick={handleGenerateSlides}
                        className="flex items-center gap-2 bg-[var(--color-primary)] hover:bg-blue-600 px-3 py-2 rounded-[var(--radius-md)] transition-colors text-white shadow-[var(--shadow-glow)] text-sm"
                    >
                        <FileText size={16} /> Slides
                    </button>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                {/* Left Column: User Notes & Timeline */}
                <div className="space-y-6">
                    <div className="glass-card p-5">
                        <h3 className="font-bold mb-4 flex items-center gap-2">
                            <FileText size={18} className="text-[var(--color-secondary)]" /> Team Notes
                        </h3>
                        <div className="mb-4">
                            <textarea
                                className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg p-3 text-sm text-white focus:border-[var(--color-primary)] outline-none min-h-[100px]"
                                placeholder="Add a note or log an update... (Markdown supported)"
                                value={newNote}
                                onChange={e => setNewNote(e.target.value)}
                            />
                            <div className="flex justify-end mt-2">
                                <button
                                    onClick={handleSaveNote}
                                    disabled={!newNote.trim() || addingNote}
                                    className="bg-[var(--color-secondary)] hover:bg-purple-600 text-white text-xs px-3 py-1.5 rounded flex items-center gap-1 disabled:opacity-50"
                                >
                                    <Plus size={14} /> Add Note
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Right Column: Performance */}
                <div className="lg:col-span-2 space-y-6">
                    <h3 className="text-xl font-semibold flex items-center gap-2">
                        <BarChart2 size={20} className="text-[var(--color-primary)]" /> Performance Data
                    </h3>

                    {/* Data Import Panel */}
                    <DataImportPanel strategyId={id} />

                    <div className="glass-panel p-8 rounded-[var(--radius-lg)] h-96 flex items-center justify-center text-[var(--color-text-muted)] bg-[rgba(0,0,0,0.2)]">
                        <div>
                            <p className="mb-2 text-center text-lg">Charts will populate here</p>
                            <p className="text-sm text-center">Comparing {strategy.monitored_urls ? strategy.monitored_urls.length : 0} URLs <br /> ({strategy.benchmark_mode})</p>
                        </div>
                    </div>
                </div>
            </div>


            {/* Timeline Section */}
            <div className="glass-card p-6">
                <h3 className="font-bold mb-6 flex items-center gap-2 text-xl">
                    <FileText size={20} className="text-[var(--color-secondary)]" /> Strategy Timeline
                </h3>
                {strategy.updates && strategy.updates.length > 0 ? (
                    <div className="border-l-2 border-[var(--color-border)] pl-6 space-y-8">
                        {strategy.updates.map((update) => (
                            <div key={update.id} className="relative group">
                                <div className="absolute -left-[29px] top-1.5 w-4 h-4 rounded-full bg-[var(--color-secondary)] border-4 border-[var(--color-bg)] group-hover:scale-110 transition-transform shadow-[0_0_10px_var(--color-secondary)]"></div>
                                <div className="flex flex-col sm:flex-row sm:items-baseline gap-2 mb-1">
                                    <span className="text-sm text-[var(--color-text-muted)] font-mono">
                                        {new Date(update.date).toLocaleString()}
                                    </span>
                                </div>
                                <div className="glass-panel p-4 rounded-lg bg-[rgba(255,255,255,0.02)]">
                                    <div className="text-sm prose prose-invert prose-sm max-w-none">
                                        <Markdown>{update.description}</Markdown>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                ) : (
                    <p className="text-[var(--color-text-muted)] italic text-center py-8">No activity recorded yet.</p>
                )}
            </div>
            <EditModal />
        </div >
    );
};

export default StrategyDetail;
