import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Save, ArrowLeft, Calendar, BarChart } from 'lucide-react';

const StrategyWizard = () => {
    const navigate = useNavigate();
    const [formData, setFormData] = useState({
        name: '',
        description: '',
        update_frequency: 'weekly',
        monitored_urls: '',
        kpis: {
            clicks: '',
            impressions: '',
            conversions: ''
        },
        collection_start_date: '',
        collection_end_date: '',
        benchmark_mode: 'prev_period'
    });
    const [loading, setLoading] = useState(false);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        try {
            const res = await axios.post('/api/strategies', formData);
            // Redirect to the detail page of the new strategy
            navigate(`/strategy/${res.data.id}`);
        } catch (err) {
            console.error(err);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="max-w-2xl mx-auto">
            <button onClick={() => navigate(-1)} className="flex items-center gap-2 text-[var(--color-text-muted)] hover:text-white mb-6">
                <ArrowLeft size={20} /> Back
            </button>

            <div className="glass-card p-8">
                <h2 className="text-2xl font-bold text-white mb-6">Create New Strategy</h2>

                <form onSubmit={handleSubmit} className="space-y-6">
                    <div>
                        <label className="block text-sm font-medium text-[var(--color-text-muted)] mb-2">Strategy Name</label>
                        <input
                            type="text"
                            required
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-3 text-white focus:border-[var(--color-primary)] outline-none"
                            placeholder="e.g. Q1 Keyword Dominance"
                            value={formData.name}
                            onChange={e => setFormData({ ...formData, name: e.target.value })}
                        />
                    </div>

                    <div>
                        <label className="block text-sm font-medium text-[var(--color-text-muted)] mb-2">Description</label>
                        <textarea
                            rows="4"
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-3 text-white focus:border-[var(--color-primary)] outline-none"
                            placeholder="Describe the goals and scope of this strategy..."
                            value={formData.description}
                            onChange={e => setFormData({ ...formData, description: e.target.value })}
                        />
                    </div>

                    <div>
                        <label className="block text-sm font-medium text-[var(--color-text-muted)] mb-2">Update Frequency</label>
                        <select
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-3 text-white focus:border-[var(--color-primary)] outline-none"
                            value={formData.update_frequency}
                            onChange={e => setFormData({ ...formData, update_frequency: e.target.value })}
                        >
                            <option value="daily">Daily</option>
                            <option value="weekly">Weekly</option>
                            <option value="monthly">Monthly</option>
                        </select>
                    </div>

                    <div>
                        <label className="block text-sm font-medium text-[var(--color-text-muted)] mb-2">Monitored URLs (one per line)</label>
                        <textarea
                            rows="5"
                            className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-3 text-white focus:border-[var(--color-primary)] outline-none font-mono text-sm"
                            placeholder="https://example.com/page1&#10;https://example.com/page2"
                            value={formData.monitored_urls}
                            onChange={e => setFormData({ ...formData, monitored_urls: e.target.value })}
                        />
                    </div>

                    <div className="p-4 border border-[var(--color-border)] rounded-lg bg-[rgba(0,0,0,0.1)]">
                        <label className="block text-sm font-bold text-white mb-4">Target KPIs (Monthly)</label>
                        <div className="grid grid-cols-3 gap-4">
                            <div>
                                <label className="block text-xs text-[var(--color-text-muted)] mb-1">Target Clicks</label>
                                <input
                                    type="number"
                                    className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none"
                                    placeholder="1000"
                                    value={formData.kpis.clicks}
                                    onChange={e => setFormData({ ...formData, kpis: { ...formData.kpis, clicks: e.target.value } })}
                                />
                            </div>
                            <div>
                                <label className="block text-xs text-[var(--color-text-muted)] mb-1">Target Impressions</label>
                                <input
                                    type="number"
                                    className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none"
                                    placeholder="50000"
                                    value={formData.kpis.impressions}
                                    onChange={e => setFormData({ ...formData, kpis: { ...formData.kpis, impressions: e.target.value } })}
                                />
                            </div>
                            <div>
                                <label className="block text-xs text-[var(--color-text-muted)] mb-1">Target Conversions</label>
                                <input
                                    type="number"
                                    className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded px-3 py-2 text-white outline-none"
                                    placeholder="50"
                                    value={formData.kpis.conversions}
                                    onChange={e => setFormData({ ...formData, kpis: { ...formData.kpis, conversions: e.target.value } })}
                                />
                            </div>
                        </div>
                    </div>

                    {/* Data Collection & Benchmarking */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="space-y-4">
                            <label className="block text-sm font-medium text-[var(--color-text-muted)] flex items-center gap-2">
                                <Calendar size={16} /> Data Collection Range (Post-Click)
                            </label>
                            <div className="grid grid-cols-2 gap-2">
                                <div>
                                    <label className="block text-xs text-[var(--color-text-muted)] mb-1">Start Date</label>
                                    <input
                                        type="date"
                                        className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-3 py-2 text-white outline-none focus:border-[var(--color-primary)]"
                                        value={formData.collection_start_date}
                                        onChange={e => setFormData({ ...formData, collection_start_date: e.target.value })}
                                        required
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs text-[var(--color-text-muted)] mb-1">End Date</label>
                                    <input
                                        type="date"
                                        className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-3 py-2 text-white outline-none focus:border-[var(--color-primary)]"
                                        value={formData.collection_end_date}
                                        onChange={e => setFormData({ ...formData, collection_end_date: e.target.value })}
                                    />
                                </div>
                            </div>
                        </div>

                        <div className="space-y-4">
                            <label className="block text-sm font-medium text-[var(--color-text-muted)] flex items-center gap-2">
                                <BarChart size={16} /> Benchmark Comparison
                            </label>
                            <div>
                                <label className="block text-xs text-[var(--color-text-muted)] mb-1">Compare Performance To:</label>
                                <select
                                    className="w-full bg-[rgba(0,0,0,0.3)] border border-[var(--color-border)] rounded-lg px-4 py-[10px] text-white focus:border-[var(--color-primary)] outline-none"
                                    value={formData.benchmark_mode}
                                    onChange={e => setFormData({ ...formData, benchmark_mode: e.target.value })}
                                >
                                    <option value="prev_period">Previous Period (MoM, WoW)</option>
                                    <option value="year_over_year">Year Over Year (YoY)</option>
                                    <option value="custom">Custom Baseline</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div className="pt-4">
                        <button
                            type="submit"
                            disabled={loading}
                            className="w-full bg-[var(--color-primary)] hover:bg-blue-600 text-white font-medium py-3 rounded-lg flex items-center justify-center gap-2 transition-colors shadow-[var(--shadow-glow)]"
                        >
                            {loading ? 'Creating...' : <><Save size={20} /> Create Strategy</>}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default StrategyWizard;
