from app import db
from datetime import datetime

class Strategy(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(128), nullable=False)
    description = db.Column(db.Text)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    update_frequency = db.Column(db.String(32), default='weekly')
    monitored_urls = db.Column(db.JSON, nullable=True) # List of specific URLs
    kpis = db.Column(db.JSON, nullable=True) # Dictionary of KPIs
    
    # New fields for data collection
    collection_start_date = db.Column(db.Date, nullable=True)
    collection_end_date = db.Column(db.Date, nullable=True)
    benchmark_mode = db.Column(db.String(32), default='prev_period') # 'prev_period', 'yoy', 'custom'
    
    integrations = db.relationship('Integration', backref='strategy', lazy='dynamic')
    updates = db.relationship('StrategyUpdate', backref='strategy', lazy='dynamic')
    performance_data = db.relationship('PerformanceData', backref='strategy', lazy='dynamic')

class Integration(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    platform = db.Column(db.String(32), nullable=False) # 'GSC' or 'GA4'
    credentials_path = db.Column(db.String(512), nullable=True) # Path to secure JSON file
    property_id = db.Column(db.String(256), nullable=True) # Site URL or Property ID
    
class StrategyUpdate(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    date = db.Column(db.DateTime, default=datetime.utcnow)
    description = db.Column(db.Text, nullable=False)
    impact_score = db.Column(db.Integer, default=0)

class PerformanceData(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    date = db.Column(db.Date, nullable=False)
    clicks = db.Column(db.Integer, default=0)
    impressions = db.Column(db.Integer, default=0)
    sessions = db.Column(db.Integer, default=0)
    conversions = db.Column(db.Integer, default=0)
    
    # Store source platform to distinguish mixed data if needed
    source = db.Column(db.String(32))

class BenchmarkData(db.Model):
    """Stores imported benchmark data from GSC/GA4 for strategy testing"""
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    platform = db.Column(db.String(32), nullable=False)  # 'GSC' or 'GA4'
    date = db.Column(db.Date, nullable=False)
    url = db.Column(db.String(512), nullable=False)
    
    # GSC Metrics
    clicks = db.Column(db.Integer, default=0)
    impressions = db.Column(db.Integer, default=0)
    ctr = db.Column(db.Float, default=0.0)
    position = db.Column(db.Float, default=0.0)
    
    # GA4 Metrics
    sessions = db.Column(db.Integer, default=0)
    pageviews = db.Column(db.Integer, default=0)
    avg_session_duration = db.Column(db.Float, default=0.0)
    bounce_rate = db.Column(db.Float, default=0.0)
    
    imported_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    __table_args__ = (
        db.Index('idx_strategy_platform_date', 'strategy_id', 'platform', 'date'),
    )

class ReportTemplate(db.Model):
    """Pre-configured report templates for common SEO analyses"""
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(128), nullable=False)
    description = db.Column(db.Text)
    category = db.Column(db.String(64))  # 'performance', 'technical', 'content', etc.
    config = db.Column(db.JSON)  # Stores report configuration
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    is_system = db.Column(db.Boolean, default=True)  # System vs user-created

class PerformanceReport(db.Model):
    """Generated performance reports with linked accounts"""
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(128), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    # Scoped OAuth connections (stored per report)
    gsc_credentials_path = db.Column(db.String(512), nullable=True)
    gsc_property_id = db.Column(db.String(256), nullable=True)
    ga4_credentials_path = db.Column(db.String(512), nullable=True)
    ga4_property_id = db.Column(db.String(256), nullable=True)
    
    # Report configuration
    urls = db.Column(db.JSON)  # List of URLs being tracked
    start_date = db.Column(db.Date)
    end_date = db.Column(db.Date)
    
    # Generated insights and recommendations
    insights = db.Column(db.Text)
    recommendations = db.Column(db.JSON)  # List of recommendation objects
    
    # Relationships
    url_metrics = db.relationship('URLMetrics', backref='report', lazy='dynamic', cascade='all, delete-orphan')

class URLMetrics(db.Model):
    """Detailed metrics for individual URLs within a performance report"""
    id = db.Column(db.Integer, primary_key=True)
    report_id = db.Column(db.Integer, db.ForeignKey('performance_report.id'), nullable=False)
    url = db.Column(db.String(512), nullable=False)
    
    # GSC Metrics
    clicks = db.Column(db.Integer, default=0)
    impressions = db.Column(db.Integer, default=0)
    ctr = db.Column(db.Float, default=0.0)
    position = db.Column(db.Float, default=0.0)
    
    # GA4 Metrics
    sessions = db.Column(db.Integer, default=0)
    pageviews = db.Column(db.Integer, default=0)
    avg_session_duration = db.Column(db.Float, default=0.0)
    bounce_rate = db.Column(db.Float, default=0.0)
    
    # CWV (Core Web Vitals)
    lcp = db.Column(db.Float, default=0.0)  # Largest Contentful Paint
    fid = db.Column(db.Float, default=0.0)  # First Input Delay
    cls = db.Column(db.Float, default=0.0)  # Cumulative Layout Shift
    
    # Rich Results & Schema
    rich_results = db.Column(db.JSON)  # Array of detected rich result types
    
    # Query data and keyword cloud
    queries = db.Column(db.JSON)  # Top queries for this URL
    keyword_cloud_path = db.Column(db.String(512))  # Path to generated word cloud image
    
    # Metadata
    publish_date = db.Column(db.Date, nullable=True)
    change_details = db.Column(db.Text) 
