import os
import google_auth_oauthlib.flow
from flask import url_for, session

# Allow OAuth over HTTP for development
os.environ['OAUTHLIB_INSECURE_TRANSPORT'] = '1'

CLIENT_SECRETS_FILE = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'client_secret.json')

# Always request all scopes to avoid "scope has changed" errors
# Users only need to grant permissions once for all platforms
ALL_SCOPES = [
    'https://www.googleapis.com/auth/webmasters.readonly',  # GSC
    'https://www.googleapis.com/auth/analytics.readonly'     # GA4
]

SCOPES = {
    'GSC': ALL_SCOPES,
    'GA4': ALL_SCOPES
}

def create_flow(platform):
    # Always use all scopes to prevent scope change errors
    scopes = ALL_SCOPES
    
    # Load .env-conf if available
    from dotenv import load_dotenv
    config_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), '.env-conf')
    load_dotenv(config_path)

    client_id = os.getenv('GOOGLE_CLIENT_ID')
    client_secret = os.getenv('GOOGLE_CLIENT_SECRET')
    redirect_uri = os.getenv('GOOGLE_REDIRECT_URI', 'https://reporting.jetlifecdn.com/v2/api/auth/google/callback')
    
    if client_id and client_secret:
        # Construct client config dictionary
        client_config = {
            "web": {
                "client_id": client_id,
                "client_secret": client_secret,
                "auth_uri": "https://accounts.google.com/o/oauth2/auth",
                "token_uri": "https://oauth2.googleapis.com/token",
                "redirect_uris": [redirect_uri]
            }
        }
        flow = google_auth_oauthlib.flow.Flow.from_client_config(
            client_config,
            scopes=scopes
        )
    else:
        # Fallback to file
        if not os.path.exists(CLIENT_SECRETS_FILE):
             raise FileNotFoundError(f"Missing OAuth credentials. Create {config_path} or client_secret.json")
             
        flow = google_auth_oauthlib.flow.Flow.from_client_secrets_file(
            CLIENT_SECRETS_FILE,
            scopes=scopes
        )
    
    # Needs to match the URI in Console
    flow.redirect_uri = redirect_uri
    
    return flow

def credentials_to_dict(credentials):
    return {
        'token': credentials.token,
        'refresh_token': credentials.refresh_token,
        'token_uri': credentials.token_uri,
        'client_id': credentials.client_id,
        'client_secret': credentials.client_secret,
        'scopes': credentials.scopes
    }
