"""
Report Templates Service
Provides pre-configured report templates for common SEO analyses
"""

REPORT_TEMPLATES = [
    {
        'id': 1,
        'name': 'Content Performance Deep Dive',
        'description': 'Comprehensive analysis of content performance including engagement, rankings, and conversion metrics',
        'category': 'performance',
        'icon': 'FileText',
        'config': {
            'metrics': [
                'clicks', 'impressions', 'ctr', 'position',
                'sessions', 'pageviews', 'bounce_rate', 'avg_session_duration',
                'conversions'
            ],
            'dimensions': ['page', 'query', 'date'],
            'analysis_types': [
                'trend_analysis',
                'query_diversity',
                'content_freshness_impact'
            ]
        },
        'insights_generated': [
            'Top performing content pieces',
            'Content gaps and opportunities',
            'Keyword diversity analysis',
            'Conversion funnel performance'
        ],
        'recommendations_provided': [
            'Content optimization priorities',
            'Internal linking opportunities',
            'Keyword targeting suggestions',
            'Conversion optimization tactics'
        ]
    },
    {
        'id': 2,
        'name': 'Technical SEO Health Check',
        'description': 'Technical performance analysis focusing on Core Web Vitals, crawlability, and indexing',
        'category': 'technical',
        'icon': 'Settings',
        'config': {
            'metrics': [
                'lcp', 'fid', 'cls',
                'crawl_errors', 'index_coverage',
                'mobile_usability'
            ],
            'analysis_types': [
                'cwv_analysis',
                'mobile_performance',
                'indexing_status'
            ]
        },
        'insights_generated': [
            'Core Web Vitals performance breakdown',
            'Mobile vs desktop performance gaps',
            'Indexing and crawl issues',
            'Page speed bottlenecks'
        ],
        'recommendations_provided': [
            'Speed optimization priorities',
            'Mobile experience improvements',
            'Technical fixes for crawl errors',
            'Schema markup implementation'
        ]
    },
    {
        'id': 3,
        'name': 'Keyword Rankings & Opportunities',
        'description': 'In-depth keyword analysis with ranking trends, query performance, and opportunity identification',
        'category': 'keywords',
        'icon': 'Hash',
        'config': {
            'metrics': [
                'query_clicks', 'query_impressions', 'query_position',
                'query_ctr'
            ],
            'dimensions': ['query', 'date', 'page'],
            'analysis_types': [
                'ranking_trends',
                'query_clustering',
                'opportunity_analysis',
                'keyword_cannibalization'
            ]
        },
        'insights_generated': [
            'Top ranking keywords',
            'Keyword ranking trends',
            'Quick win opportunities (position 11-20)',
            'Keyword cannibalization issues',
            'Long-tail keyword performance'
        ],
        'recommendations_provided': [
            'Priority keywords to optimize',
            'Content consolidation for cannibalization',
            'New content opportunities',
            'Keyword targeting improvements'
        ]
    },
    {
        'id': 4,
        'name': 'Conversion Path Analysis',
        'description': 'Track user journey from organic search to conversion with attribution insights',
        'category': 'conversion',
        'icon': 'TrendingUp',
        'config': {
            'metrics': [
                'sessions', 'conversions', 'conversion_rate',
                'avg_session_duration', 'pages_per_session',
                'bounce_rate'
            ],
            'dimensions': ['landing_page', 'source_medium', 'device'],
            'analysis_types': [
                'conversion_funnel',
                'landing_page_performance',
                'device_attribution'
            ]
        },
        'insights_generated': [
            'Top converting landing pages',
            'Conversion rate by device',
            'User engagement metrics',
            'Drop-off points in funnel'
        ],
        'recommendations_provided': [
            'Landing page optimization tactics',
            'CTA placement improvements',
            'Mobile conversion optimization',
            'Trust signal enhancements'
        ]
    },
    {
        'id': 5,
        'name': 'Competitive Visibility Analysis',
        'description': 'Analyze search visibility across queries and identify competitive gaps',
        'category': 'competitive',
        'icon': 'Target',
        'config': {
            'metrics': [
                'impressions', 'position', 'share_of_voice'
            ],
            'dimensions': ['query', 'page'],
            'analysis_types': [
                'visibility_trends',
                'query_market_share',
                'position_distribution'
            ]
        },
        'insights_generated': [
            'Search visibility trends',
            'Position distribution analysis',
            'High-impression low-CTR queries',
            'Visibility gaps by topic'
        ],
        'recommendations_provided': [
            'Content topics to cover',
            'Title and meta improvements for visibility',
            'Featured snippet opportunities',
            'Priority pages for optimization'
        ]
    },
    {
        'id': 6,
        'name': 'Rich Results Performance',
        'description': 'Analyze rich result performance and identify structured data opportunities',
        'category': 'rich_results',
        'icon': 'Sparkles',
        'config': {
            'metrics': [
                'rich_result_impressions', 'rich_result_clicks',
                'rich_result_ctr'
            ],
            'dimensions': ['search_appearance', 'page'],
            'analysis_types': [
                'rich_results_coverage',
                'rich_results_performance',
                'schema_opportunities'
            ]
        },
        'insights_generated': [
            'Current rich result coverage',
            'Rich result CTR vs standard results',
            'Most effective rich result types',
            'Pages without rich results'
        ],
        'recommendations_provided': [
            'Schema markup to implement',
            'Rich result optimization tactics',
            'FAQ and How-to opportunities',
            'Product schema enhancements'
        ]
    },
    {
        'id': 7,
        'name': 'Content Refresh ROI',
        'description': 'Measure impact of content updates and optimizations on performance',
        'category': 'content',
        'icon': 'RefreshCw',
        'config': {
            'metrics': [
                'clicks', 'impressions', 'position', 'ctr',
                'sessions', 'conversions'
            ],
            'dimensions': ['page', 'date'],
            'analysis_types': [
                'before_after_comparison',
                'trend_analysis',
                'impact_quantification'
            ],
            'requires_publish_dates': True
        },
        'insights_generated': [
            'Performance change after updates',
            'Time to impact measurement',
            'ROI of content refreshes',
            'Which updates drove results'
        ],
        'recommendations_provided': [
            'Which content to refresh next',
            'Content update best practices',
            'Optimization tactics that worked',
            'Update frequency recommendations'
        ]
    },
    {
        'id': 8,
        'name': 'Mobile SEO Performance',
        'description': 'Mobile-specific analysis including usability, speed, and ranking differences',
        'category': 'mobile',
        'icon': 'Smartphone',
        'config': {
            'metrics': [
                'mobile_clicks', 'mobile_impressions', 'mobile_ctr',
                'mobile_position', 'mobile_usability_issues',
                'lcp_mobile', 'cls_mobile'
            ],
            'dimensions': ['device', 'page'],
            'analysis_types': [
                'mobile_vs_desktop',
                'mobile_usability',
                'mobile_speed'
            ]
        },
        'insights_generated': [
            'Mobile vs desktop performance gaps',
            'Mobile usability issues',
            'Mobile Core Web Vitals',
            'Mobile-specific ranking factors'
        ],
        'recommendations_provided': [
            'Mobile speed optimizations',
            'Touch target improvements',
            'Mobile UX enhancements',
            'Responsive design fixes'
        ]
    },
    {
        'id': 9,
        'name': 'Local SEO Dashboard',
        'description': 'Local search performance including maps, local pack, and geo-specific queries',
        'category': 'local',
        'icon': 'MapPin',
        'config': {
            'metrics': [
                'local_clicks', 'local_impressions',
                'maps_views', 'direction_requests'
            ],
            'dimensions': ['query', 'page', 'location'],
            'analysis_types': [
                'local_visibility',
                'geo_performance',
                'maps_engagement'
            ]
        },
        'insights_generated': [
            'Local search visibility',
            'Top local queries',
            'Geographic performance distribution',
            'Maps vs web search performance'
        ],
        'recommendations_provided': [
            'Google Business Profile optimizations',
            'Local content opportunities',
            'Citation building priorities',
            'Location page improvements'
        ]
    },
    {
        'id': 10,
        'name': 'Executive SEO Summary',
        'description': 'High-level strategic overview with KPIs and business impact for stakeholder reporting',
        'category': 'executive',
        'icon': 'BarChart3',
        'config': {
            'metrics': [
                'organic_traffic', 'conversions', 'revenue',
                'visibility_score', 'avg_position'
            ],
            'dimensions': ['date'],
            'analysis_types': [
                'kpi_tracking',
                'trend_analysis',
                'business_impact'
            ],
            'presentation_mode': 'executive'
        },
        'insights_generated': [
            'Key performance indicators',
            'Month-over-month trends',
            'Business impact metrics',
            'Strategic wins and challenges'
        ],
        'recommendations_provided': [
            'Strategic priorities',
            'Investment recommendations',
            'Resource allocation suggestions',
            'Competitive positioning'
        ]
    }
]


def get_all_templates():
    """Get all available report templates"""
    return REPORT_TEMPLATES


def get_template_by_id(template_id):
    """Get specific template by ID"""
    for template in REPORT_TEMPLATES:
        if template['id'] == template_id:
            return template
    return None


def get_templates_by_category(category):
    """Get templates filtered by category"""
    return [t for t in REPORT_TEMPLATES if t['category'] == category]


def get_template_categories():
    """Get unique categories"""
    categories = {}
    for template in REPORT_TEMPLATES:
        cat = template['category']
        if cat not in categories:
            categories[cat] = {
                'key': cat,
                'label': cat.replace('_', ' ').title(),
                'count': 0
            }
        categories[cat]['count'] += 1
    return list(categories.values())
