from flask import Blueprint, jsonify, current_app
from datetime import datetime

main = Blueprint('main', __name__)

@main.route('/api/health')
def health_check():
    return jsonify({'status': 'ok', 'message': 'V2 SEO App Backend Running'})

# --- Strategy Routes ---

from app import db
from app.models import Strategy, StrategyUpdate
from flask import request

@main.route('/api/strategies', methods=['GET'])
def get_strategies():
    strategies = Strategy.query.order_by(Strategy.created_at.desc()).all()
    return jsonify([{
        'id': s.id,
        'name': s.name,
        'description': s.description,
        'update_frequency': s.update_frequency,
        'created_at': s.created_at.isoformat()
    } for s in strategies])

@main.route('/api/strategies', methods=['POST'])
def create_strategy():
    data = request.json
    
    # Parse monitored URLs (split by newlines if string)
    urls = data.get('monitored_urls', [])
    if isinstance(urls, str):
        urls = [u.strip() for u in urls.split('\n') if u.strip()]
        
    kpis = data.get('kpis', {})
    
    new_strategy = Strategy(
        name=data.get('name'),
        description=data.get('description'),
        update_frequency=data.get('update_frequency', 'weekly'),
        monitored_urls=urls,
        kpis=kpis,
        collection_start_date=datetime.fromisoformat(data['collection_start_date']).date() if data.get('collection_start_date') else None,
        collection_end_date=datetime.fromisoformat(data['collection_end_date']).date() if data.get('collection_end_date') else None,
        benchmark_mode=data.get('benchmark_mode', 'prev_period')
    )
    db.session.add(new_strategy)
    db.session.commit()
    return jsonify({'message': 'Strategy created', 'id': new_strategy.id}), 201

@main.route('/api/strategies/<int:id>', methods=['GET'])
def get_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    return jsonify({
        'id': strategy.id,
        'name': strategy.name,
        'description': strategy.description,
        'update_frequency': strategy.update_frequency,
        'created_at': strategy.created_at.isoformat(),
        'monitored_urls': strategy.monitored_urls,
        'kpis': strategy.kpis,
        'integrations': [{'platform': i.platform, 'connected': True} for i in strategy.integrations],
        'updates': [{
            'id': u.id,
            'date': u.date.isoformat(),
            'description': u.description,
            'impact_score': u.impact_score
        } for u in strategy.updates.order_by(StrategyUpdate.date.desc()).all()]
    })

@main.route('/api/strategies/<int:id>/updates', methods=['POST'])
def create_strategy_update(id):
    strategy = Strategy.query.get_or_404(id)
    data = request.json
    
    update = StrategyUpdate(
        strategy_id=id,
        description=data.get('description'),
        impact_score=data.get('impact_score', 0)
    )
    db.session.add(update)
    db.session.commit()
    
    return jsonify({
        'message': 'Update added',
        'update': {
            'id': update.id,
            'date': update.date.isoformat(),
            'description': update.description,
            'impact_score': update.impact_score
        }
    }), 201

@main.route('/api/strategies/<int:id>', methods=['PUT'])
def update_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    data = request.json
    changes = []
    
    if 'name' in data and data['name'] != strategy.name:
        changes.append(f"Name updated to '{data['name']}'")
        strategy.name = data['name']
        
    if 'description' in data and data['description'] != strategy.description:
        changes.append("Description updated")
        strategy.description = data['description']
        
    if 'update_frequency' in data and data['update_frequency'] != strategy.update_frequency:
        changes.append(f"Frequency changed to {data['update_frequency']}")
        strategy.update_frequency = data['update_frequency']
    
    # Update URLs if provided
    urls = data.get('monitored_urls')
    if urls is not None:
        if isinstance(urls, str):
            urls = [u.strip() for u in urls.split('\n') if u.strip()]
        
        # Compare lists (simplified)
        if urls != strategy.monitored_urls:
            count_diff = len(urls) - (len(strategy.monitored_urls) if strategy.monitored_urls else 0)
            changes.append(f"Monitored URLs updated ({len(urls)} total)")
        strategy.monitored_urls = urls
        
    # Update KPIs if provided
    kpis = data.get('kpis')
    if kpis is not None:
        if kpis != strategy.kpis:
             changes.append("KPI targets updated")
        strategy.kpis = kpis
        
    if changes:
        update_desc = "**Settings Updated:** " + "; ".join(changes)
        new_update = StrategyUpdate(
            strategy_id=id,
            description=update_desc,
            impact_score=0
        )
        db.session.add(new_update)
        
    db.session.commit()
    return jsonify({'message': 'Strategy updated'})

@main.route('/api/strategies/<int:id>', methods=['DELETE'])
def delete_strategy(id):
    strategy = Strategy.query.get_or_404(id)
    db.session.delete(strategy)
    db.session.commit()
    return jsonify({'message': 'Strategy deleted'})

@main.route('/api/strategies/<int:id>/import-data', methods=['POST'])
def import_benchmark_data(id):
    """Import benchmark data from GSC or GA4 for a strategy"""
    from app.models import BenchmarkData
    from app.performance_service import PerformanceDataService
    from datetime import datetime as dt, timedelta
    
    strategy = Strategy.query.get_or_404(id)
    data = request.json
    platform = data.get('platform')  # 'GSC' or 'GA4'
    date_range = data.get('dateRange', {})
    
    start_date = date_range.get('start')
    end_date = date_range.get('end')
    
    if not platform or platform not in ['GSC', 'GA4']:
        return jsonify({'error': 'Invalid platform'}), 400
    
    if not start_date or not end_date:
        return jsonify({'error': 'Date range required'}), 400
    
    # Find the integration for this strategy and platform
    integration = Integration.query.filter_by(strategy_id=id, platform=platform).first()
    
    if not integration or not integration.credentials_path:
        return jsonify({'error': f'{platform} not connected for this strategy'}), 400
    
    # Get monitored URLs from strategy
    urls = strategy.monitored_urls or []
    if not urls:
        return jsonify({'error': 'No monitored URLs configured for this strategy'}), 400
    
    try:
        # Initialize service
        service = PerformanceDataService(current_app.instance_path)
        
        # Fetch data based on platform
        if platform == 'GSC':
            # Determine property_id (site URL)
            property_id = integration.property_id or 'sc-domain:example.com'  # Fallback
            
            # Fetch day-by-day data
            result_data = []
            current_date = dt.fromisoformat(start_date).date()
            end_dt = dt.fromisoformat(end_date).date()
            
            while current_date <= end_dt:
                date_str = current_date.isoformat()
                
                # Fetch data for this specific day
                daily_data = service.fetch_gsc_enhanced_data(
                    integration.credentials_path,
                    property_id,
                    urls,
                    date_str,
                    date_str
                )
                
                # Store in database
                for url, metrics in daily_data.items():
                    benchmark = BenchmarkData(
                        strategy_id=id,
                        platform=platform,
                        date=current_date,
                        url=url,
                        clicks=metrics.get('clicks', 0),
                        impressions=metrics.get('impressions', 0),
                        ctr=metrics.get('ctr', 0) / 100 if metrics.get('ctr') else 0,  # Convert to decimal
                        position=metrics.get('position', 0)
                    )
                    db.session.add(benchmark)
                    
                    # Add to result for frontend
                    result_data.append({
                        'date': date_str,
                        'url': url,
                        'clicks': metrics.get('clicks', 0),
                        'impressions': metrics.get('impressions', 0),
                        'ctr': metrics.get('ctr', 0) / 100 if metrics.get('ctr') else 0,
                        'position': metrics.get('position', 0)
                    })
                
                current_date += timedelta(days=1)
            
            db.session.commit()
            return jsonify({'message': 'GSC data imported successfully', 'data': result_data})
            
        elif platform == 'GA4':
            property_id = integration.property_id or '123456789'  # Fallback
            
            # Fetch day-by-day data
            result_data = []
            current_date = dt.fromisoformat(start_date).date()
            end_dt = dt.fromisoformat(end_date).date()
            
            while current_date <= end_dt:
                date_str = current_date.isoformat()
                
                # Fetch data for this specific day
                daily_data = service.fetch_ga4_enhanced_data(
                    integration.credentials_path,
                    property_id,
                    urls,
                    date_str,
                    date_str
                )
                
                # Store in database
                for url, metrics in daily_data.items():
                    benchmark = BenchmarkData(
                        strategy_id=id,
                        platform=platform,
                        date=current_date,
                        url=url,
                        sessions=metrics.get('sessions', 0),
                        pageviews=metrics.get('pageviews', 0),
                        avg_session_duration=metrics.get('avg_session_duration', 0),
                        bounce_rate=metrics.get('bounce_rate', 0)
                    )
                    db.session.add(benchmark)
                    
                    # Add to result for frontend
                    result_data.append({
                        'date': date_str,
                        'url': url,
                        'sessions': metrics.get('sessions', 0),
                        'pageviews': metrics.get('pageviews', 0),
                        'avgSessionDuration': metrics.get('avg_session_duration', 0),
                        'bounceRate': metrics.get('bounce_rate', 0)
                    })
                
                current_date += timedelta(days=1)
            
            db.session.commit()
            return jsonify({'message': 'GA4 data imported successfully', 'data': result_data})
            
    except Exception as e:
        current_app.logger.error(f'Import error: {e}')
        import traceback
        traceback.print_exc()
        return jsonify({'error': str(e)}), 500



# --- OAuth Routes ---

from flask import redirect, session, url_for
from app.oauth_manager import create_flow, credentials_to_dict
from app.models import Integration

@main.route('/api/auth/google/login/<int:strategy_id>')
def login(strategy_id):
    platform = request.args.get('platform', 'GSC')
    
    # Check if strategy exists
    strategy = Strategy.query.get_or_404(strategy_id)
    
    # Create flow
    try:
        flow = create_flow(platform)
        authorization_url, state = flow.authorization_url(
            access_type='offline',
            include_granted_scopes='true'
        )
    except Exception as e:
        return jsonify({'error': f'OAuth Error: {str(e)}'}), 500
    
    # Store state and context in session
    session['oauth_state'] = state
    session['oauth_strategy_id'] = strategy_id
    session['oauth_platform'] = platform
    
    return redirect(authorization_url)

@main.route('/api/auth/google/callback')
def callback():
    state = session.get('oauth_state')
    # In production, check state explicitly against request.args['state']
    
    flow = create_flow(session.get('oauth_platform'))
    try:
        flow.fetch_token(authorization_response=request.url)
    except Exception as e:
        return jsonify({'error': str(e)}), 400
    
    credentials = flow.credentials
    creds_dict = credentials_to_dict(credentials)
    
    strategy_id = session.get('oauth_strategy_id')
    platform = session.get('oauth_platform')
    
    # Save to DB
    # Check if integration exists
    integration = Integration.query.filter_by(strategy_id=strategy_id, platform=platform).first()
    if not integration:
        integration = Integration(strategy_id=strategy_id, platform=platform)
        db.session.add(integration)
        
    import json
    import os
    
    # Save to File in instance/tokens/
    token_dir = os.path.join(current_app.instance_path, 'tokens')
    os.makedirs(token_dir, exist_ok=True)
    
    filename = f"strategy_{strategy_id}_{platform}.json"
    file_path = os.path.join(token_dir, filename)
    
    with open(file_path, 'w') as f:
        json.dump(creds_dict, f)
        
    integration.credentials_path = file_path
    
    # Simple logic to guess property_id if possible, or leave null for user to select later
    # For now, just connected
    
    db.session.commit()
    
    # Redirect back to frontend
    return redirect(f'/v2/strategy/{strategy_id}?connected={platform}')


from app.tasks import generate_presentation
from flask import send_file

@main.route('/api/strategies/<int:id>/generate-slides', methods=['POST'])
def generate_slides(id):
    try:
        path = generate_presentation(id)
        if not path:
            return jsonify({'error': 'Strategy not found'}), 404
            
        return send_file(path, as_attachment=True, download_name=f"strategy_{id}.pptx")
    except Exception as e:
        return jsonify({'error': str(e)}), 500



import shutil
from app.models import Strategy, Integration

@main.route('/api/system/stats')
def system_stats():
    total, used, free = shutil.disk_usage("/")
    
    strategy_count = Strategy.query.count()
    integration_count = Integration.query.count()
    
    return jsonify({
        'storage': {
            'total': total,
            'used': used,
            'free': free,
            'percent': round((used / total) * 100, 1)
        },
        'metrics': {
            'strategies': strategy_count,
            'integrations': integration_count,
            'tests_tracked': strategy_count * 5 # Placeholder logic
        }
    })

# --- Performance Reports Routes ---

from datetime import datetime, timedelta
import json
import os

@main.route('/api/performance-reports/fetch', methods=['POST'])
def fetch_performance_data():
    """Fetch GA4 and GSC data for specific URLs in a date range"""
    data = request.json
    urls = data.get('urls', [])
    start_date = data.get('startDate')
    end_date = data.get('endDate')
    publish_dates = data.get('publishDates', {})
    
    if not urls or not start_date or not end_date:
        return jsonify({'error': 'Missing required parameters'}), 400
    
    # Initialize result structure
    result = {
        'byUrl': {},
        'totals': {
            'clicks': 0,
            'impressions': 0,
            'sessions': 0,
            'conversions': 0,
            'ctr': 0
        },
        'timeframe': {
            'start': start_date,
            'end': end_date
        }
    }
    
    try:
        # Look for any connected integration to get credentials
        # In a real scenario, you'd want to specify which strategy's credentials to use
        integration_gsc = Integration.query.filter_by(platform='GSC').first()
        integration_ga4 = Integration.query.filter_by(platform='GA4').first()
        
        if not integration_gsc and not integration_ga4:
            return jsonify({'error': 'No GSC or GA4 integration found. Please connect at least one service.'}), 400
        
        # Fetch GSC data if available
        gsc_data = {}
        if integration_gsc and integration_gsc.credentials_path:
            try:
                gsc_data = fetch_gsc_data(integration_gsc.credentials_path, urls, start_date, end_date)
            except Exception as e:
                current_app.logger.error(f"GSC fetch error: {e}")
        
        # Fetch GA4 data if available
        ga4_data = {}
        if integration_ga4 and integration_ga4.credentials_path:
            try:
                ga4_data = fetch_ga4_data(integration_ga4.credentials_path, urls, start_date, end_date)
            except Exception as e:
                current_app.logger.error(f"GA4 fetch error: {e}")
        
        # Merge data by URL
        for url in urls:
            url_data = {
                'clicks': gsc_data.get(url, {}).get('clicks', 0),
                'impressions': gsc_data.get(url, {}).get('impressions', 0),
                'sessions': ga4_data.get(url, {}).get('sessions', 0),
                'conversions': ga4_data.get(url, {}).get('conversions', 0),
                'ctr': 0
            }
            
            # Calculate CTR
            if url_data['impressions'] > 0:
                url_data['ctr'] = (url_data['clicks'] / url_data['impressions']) * 100
            
            result['byUrl'][url] = url_data
            
            # Update totals
            result['totals']['clicks'] += url_data['clicks']
            result['totals']['impressions'] += url_data['impressions']
            result['totals']['sessions'] += url_data['sessions']
            result['totals']['conversions'] += url_data['conversions']
        
        # Calculate overall CTR
        if result['totals']['impressions'] > 0:
            result['totals']['ctr'] = (result['totals']['clicks'] / result['totals']['impressions']) * 100
        
        return jsonify(result)
        
    except Exception as e:
        current_app.logger.error(f"Error fetching performance data: {e}")
        return jsonify({'error': str(e)}), 500


def fetch_gsc_data(credentials_path, urls, start_date, end_date):
    """Fetch Google Search Console data for specific URLs"""
    from google.oauth2.credentials import Credentials
    from googleapiclient.discovery import build
    
    # Load credentials
    with open(credentials_path, 'r') as f:
        creds_dict = json.load(f)
    
    creds = Credentials(
        token=creds_dict['token'],
        refresh_token=creds_dict.get('refresh_token'),
        token_uri=creds_dict['token_uri'],
        client_id=creds_dict['client_id'],
        client_secret=creds_dict['client_secret'],
        scopes=creds_dict['scopes']
    )
    
    service = build('webmasters', 'v3', credentials=creds)
    
    # You would need to know the site URL from the integration
    # For now, using a placeholder - this should be stored in Integration.property_id
    site_url = 'sc-domain:example.com'  # This should come from the integration
    
    result = {}
    
    for url in urls:
        try:
            response = service.searchanalytics().query(
                siteUrl=site_url,
                body={
                    'startDate': start_date,
                    'endDate': end_date,
                    'dimensions': ['page'],
                    'dimensionFilterGroups': [{
                        'filters': [{
                            'dimension': 'page',
                            'operator': 'equals',
                            'expression': url
                        }]
                    }]
                }
            ).execute()
            
            if 'rows' in response and len(response['rows']) > 0:
                row = response['rows'][0]
                result[url] = {
                    'clicks': row.get('clicks', 0),
                    'impressions': row.get('impressions', 0),
                    'ctr': row.get('ctr', 0) * 100,
                    'position': row.get('position', 0)
                }
            else:
                result[url] = {'clicks': 0, 'impressions': 0, 'ctr': 0, 'position': 0}
                
        except Exception as e:
            current_app.logger.error(f"Error fetching GSC data for {url}: {e}")
            result[url] = {'clicks': 0, 'impressions': 0, 'ctr': 0, 'position': 0}
    
    return result


def fetch_ga4_data(credentials_path, urls, start_date, end_date):
    """Fetch Google Analytics 4 data for specific URLs"""
    from google.oauth2.credentials import Credentials
    from googleapiclient.discovery import build
    
    # Load credentials
    with open(credentials_path, 'r') as f:
        creds_dict = json.load(f)
    
    creds = Credentials(
        token=creds_dict['token'],
        refresh_token=creds_dict.get('refresh_token'),
        token_uri=creds_dict['token_uri'],
        client_id=creds_dict['client_id'],
        client_secret=creds_dict['client_secret'],
        scopes=creds_dict['scopes']
    )
    
    # Use GA4 Data API
    from google.analytics.data_v1beta import BetaAnalyticsDataClient
    from google.analytics.data_v1beta.types import RunReportRequest, DateRange, Dimension, Metric, FilterExpression, Filter
    
    # This requires google-analytics-data package
    # Property ID should come from Integration.property_id
    property_id = '123456789'  # Placeholder
    
    result = {}
    
    # For now, return mock data since GA4 API setup is complex
    # In production, you'd use the BetaAnalyticsDataClient with proper credentials
    for url in urls:
        result[url] = {
            'sessions': 0,
            'conversions': 0,
            'pageviews': 0,
            'avgSessionDuration': 0
        }
    
    return result


@main.route('/api/performance-reports/generate-insights', methods=['POST'])
def generate_performance_insights():
    """Generate AI-powered insights from performance data"""
    data = request.json
    performance_data = data.get('performanceData', {})
    rows = data.get('rows', [])
    
    if not performance_data:
        return jsonify({'error': 'No performance data provided'}), 400
    
    # Build a summary of the data
    totals = performance_data.get('totals', {})
    by_url = performance_data.get('byUrl', {})
    
    insights = []
    
    # Overall performance summary
    insights.append(f"## Performance Summary")
    insights.append(f"")
    insights.append(f"Between {performance_data['timeframe']['start']} and {performance_data['timeframe']['end']}:")
    insights.append(f"- **Total Clicks:** {totals['clicks']:,}")
    insights.append(f"- **Total Impressions:** {totals['impressions']:,}")
    insights.append(f"- **Total Sessions:** {totals['sessions']:,}")
    insights.append(f"- **Average CTR:** {totals['ctr']:.2f}%")
    insights.append(f"")
    
    # Per-URL insights
    insights.append(f"## URL Performance Breakdown")
    insights.append(f"")
    
    # Sort URLs by clicks (descending)
    sorted_urls = sorted(by_url.items(), key=lambda x: x[1].get('clicks', 0), reverse=True)
    
    for i, (url, data) in enumerate(sorted_urls[:5], 1):  # Top 5 URLs
        # Find the row data for context
        row_data = next((r for r in rows if r.get('url') == url), {})
        details = row_data.get('details', 'No change details provided')
        publish_date = row_data.get('publishDate', 'Unknown date')
        
        insights.append(f"### {i}. {url}")
        insights.append(f"**Published:** {publish_date} | **Change:** {details}")
        insights.append(f"- Clicks: {data['clicks']:,} | Impressions: {data['impressions']:,} | CTR: {data.get('ctr', 0):.2f}%")
        insights.append(f"")
    
    # Generate actionable insights
    insights.append(f"## Key Insights & Recommendations")
    insights.append(f"")
    
    if totals['ctr'] < 2.0:
        insights.append(f"⚠️ **Low CTR Alert:** Your average CTR of {totals['ctr']:.2f}% is below industry benchmarks (2-3%). Consider:")
        insights.append(f"  - Optimizing meta titles and descriptions")
        insights.append(f"  - Adding rich snippets or schema markup")
        insights.append(f"  - Testing different title formulations")
        insights.append(f"")
    elif totals['ctr'] > 5.0:
        insights.append(f"✅ **Excellent CTR:** Your {totals['ctr']:.2f}% CTR is above average! This indicates strong search appeal.")
        insights.append(f"")
    
    if totals['impressions'] > 0 and totals['clicks'] / totals['impressions'] > 0.05:
        insights.append(f"📈 **High Engagement Detected:** Strong click-through rate suggests content resonates with searchers.")
        insights.append(f"")
    
    # Check for recent publications
    recent_changes = [r for r in rows if r.get('publishDate') and r.get('details')]
    if recent_changes:
        insights.append(f"🔄 **Recent Changes Tracked:** {len(recent_changes)} URLs with documented changes.")
        insights.append(f"  Monitor these closely over the next 2-4 weeks to measure impact.")
        insights.append(f"")
    
    return jsonify({'insights': '\n'.join(insights)})


# --- Enhanced Performance Reports with Scoped OAuth ---

from app.models import PerformanceReport, URLMetrics
from app.performance_service import PerformanceDataService
from datetime import date as date_class

@main.route('/api/performance-reports', methods=['GET'])
def get_performance_reports():
    """Get all performance reports"""
    reports = PerformanceReport.query.order_by(PerformanceReport.created_at.desc()).all()
    return jsonify([{
        'id': r.id,
        'name': r.name,
        'created_at': r.created_at.isoformat(),
        'start_date': r.start_date.isoformat() if r.start_date else None,
        'end_date': r.end_date.isoformat() if r.end_date else None,
        'url_count': len(r.urls) if r.urls else 0,
        'has_gsc': bool(r.gsc_credentials_path),
        'has_ga4': bool(r.ga4_credentials_path)
    } for r in reports])

@main.route('/api/performance-reports', methods=['POST'])
def create_performance_report():
    """Create a new performance report"""
    data = request.json
    
    report = PerformanceReport(
        name=data.get('name', f"Performance Report - {datetime.now().strftime('%Y-%m-%d')}"),
        urls=data.get('urls', []),
        start_date=datetime.fromisoformat(data['start_date']).date() if data.get('start_date') else None,
        end_date=datetime.fromisoformat(data['end_date']).date() if data.get('end_date') else None
    )
    
    db.session.add(report)
    db.session.commit()
    
    return jsonify({
        'message': 'Performance report created',
        'id': report.id,
        'redirect_url': f'/v2/performance-reports/{report.id}'
    }), 201

@main.route('/api/performance-reports/<int:report_id>', methods=['GET'])
def get_performance_report(report_id):
    """Get a specific performance report with all data"""
    report = PerformanceReport.query.get_or_404(report_id)
    
    # Get URL metrics
    url_metrics = URLMetrics.query.filter_by(report_id=report_id).all()
    
    return jsonify({
        'id': report.id,
        'name': report.name,
        'created_at': report.created_at.isoformat(),
        'start_date': report.start_date.isoformat() if report.start_date else None,
        'end_date': report.end_date.isoformat() if report.end_date else None,
        'urls': report.urls,
        'has_gsc': bool(report.gsc_credentials_path),
        'has_ga4': bool(report.ga4_credentials_path),
        'gsc_property_id': report.gsc_property_id,
        'ga4_property_id': report.ga4_property_id,
        'insights': report.insights,
        'recommendations': report.recommendations,
        'url_metrics': [{
            'id': um.id,
            'url': um.url,
            'clicks': um.clicks,
            'impressions': um.impressions,
            'ctr': um.ctr,
            'position': um.position,
            'sessions': um.sessions,
            'pageviews': um.pageviews,
            'avg_session_duration': um.avg_session_duration,
            'bounce_rate': um.bounce_rate,
            'cwv': {
                'lcp': um.lcp,
                'fid': um.fid,
                'cls': um.cls
            },
            'rich_results': um.rich_results,
            'queries_count': len(um.queries) if um.queries else 0,
            'queries': um.queries[:10] if um.queries else [],  # Top 10 for initial view
            'keyword_cloud_path': um.keyword_cloud_path,
            'publish_date': um.publish_date.isoformat() if um.publish_date else None,
            'change_details': um.change_details
        } for um in url_metrics]
    })

@main.route('/api/performance-reports/<int:report_id>/connect-oauth', methods=['POST'])
def connect_report_oauth(report_id):
    """Initiate OAuth flow for connecting GSC or GA4 to a specific report"""
    report = PerformanceReport.query.get_or_404(report_id)
    platform = request.json.get('platform')  # 'GSC' or 'GA4'
    
    if platform not in ['GSC', 'GA4']:
        return jsonify({'error': 'Invalid platform'}), 400
    
    try:
        flow = create_flow(platform)
        authorization_url, state = flow.authorization_url(
            access_type='offline',
            include_granted_scopes='true'
        )
    except Exception as e:
        return jsonify({'error': f'OAuth Error: {str(e)}'}), 500
    
    # Store state and context in session
    session['oauth_state'] = state
    session['oauth_report_id'] = report_id
    session['oauth_platform'] = platform
    session['oauth_context'] = 'performance_report'
    
    return jsonify({'authorization_url': authorization_url})

@main.route('/api/performance-reports/oauth-callback')
def performance_report_oauth_callback():
    """OAuth callback for performance report connections"""
    state = session.get('oauth_state')
    report_id = session.get('oauth_report_id')
    platform = session.get('oauth_platform')
    
    if not report_id or session.get('oauth_context') != 'performance_report':
        return jsonify({'error': 'Invalid OAuth session'}), 400
    
    report = PerformanceReport.query.get_or_404(report_id)
    
    flow = create_flow(platform)
    try:
        flow.fetch_token(authorization_response=request.url)
    except Exception as e:
        return jsonify({'error': str(e)}), 400
    
    credentials = flow.credentials
    creds_dict = credentials_to_dict(credentials)
    
    # Save credentials to file
    token_dir = os.path.join(current_app.instance_path, 'tokens', 'performance_reports')
    os.makedirs(token_dir, exist_ok=True)
    
    filename = f"report_{report_id}_{platform}.json"
    file_path = os.path.join(token_dir, filename)
    
    with open(file_path, 'w') as f:
        json.dump(creds_dict, f)
    
    # Update report with credentials
    if platform == 'GSC':
        report.gsc_credentials_path = file_path
    else:  # GA4
        report.ga4_credentials_path = file_path
    
    db.session.commit()
    
    # Redirect back to performance report page
    return redirect(f'/v2/performance-reports/{report_id}?connected={platform}')

@main.route('/api/performance-reports/<int:report_id>/set-property', methods=['POST'])
def set_report_property(report_id):
    """Set GSC site URL or GA4 property ID for a report"""
    report = PerformanceReport.query.get_or_404(report_id)
    data = request.json
    
    platform = data.get('platform')
    property_id = data.get('property_id')
    
    if platform == 'GSC':
        report.gsc_property_id = property_id
    elif platform == 'GA4':
        report.ga4_property_id = property_id
    else:
        return jsonify({'error': 'Invalid platform'}), 400
    
    db.session.commit()
    return jsonify({'message': 'Property ID set successfully'})

@main.route('/api/performance-reports/<int:report_id>/fetch-data', methods=['POST'])
def fetch_enhanced_performance_data(report_id):
    """
    Fetch comprehensive performance data including:
    - GSC: clicks, impressions, CTR, position, queries, rich results
    - GA4: sessions, pageviews, CWV, bounce rate
    """
    report = PerformanceReport.query.get_or_404(report_id)
    data = request.json
    
    # Get configuration from request or use report defaults
    urls = data.get('urls', report.urls)
    start_date = data.get('start_date', report.start_date.isoformat() if report.start_date else None)
    end_date = data.get('end_date', report.end_date.isoformat() if report.end_date else None)
    url_metadata = data.get('url_metadata', {})  # publish_date, change_details
    
    if not urls or not start_date or not end_date:
        return jsonify({'error': 'Missing required parameters'}), 400
    
    # Update report if needed
    if data.get('urls'):
        report.urls = urls
    if data.get('start_date'):
        report.start_date = datetime.fromisoformat(start_date).date()
    if data.get('end_date'):
        report.end_date = datetime.fromisoformat(end_date).date()
    
    # Initialize performance service
    service = PerformanceDataService(current_app.instance_path)
    
    # Fetch GSC data
    gsc_data = {}
    if report.gsc_credentials_path and report.gsc_property_id:
        try:
            gsc_data = service.fetch_gsc_enhanced_data(
                report.gsc_credentials_path,
                report.gsc_property_id,
                urls,
                start_date,
                end_date
            )
        except Exception as e:
            current_app.logger.error(f"GSC fetch error: {e}")
            return jsonify({'error': f'GSC Error: {str(e)}'}), 500
    
    # Fetch GA4 data
    ga4_data = {}
    if report.ga4_credentials_path and report.ga4_property_id:
        try:
            ga4_data = service.fetch_ga4_enhanced_data(
                report.ga4_credentials_path,
                report.ga4_property_id,
                urls,
                start_date,
                end_date
            )
        except Exception as e:
            current_app.logger.error(f"GA4 fetch error: {e}")
            return jsonify({'error': f'GA4 Error: {str(e)}'}), 500
    
    # Delete old URL metrics for this report
    URLMetrics.query.filter_by(report_id=report_id).delete()
    
    # Create URL metrics
    url_metrics_list = []
    for url in urls:
        gsc = gsc_data.get(url, {})
        ga4 = ga4_data.get(url, {})
        metadata = url_metadata.get(url, {})
        
        # Generate keyword cloud
        queries = gsc.get('queries', [])
        keyword_cloud_path = None
        if queries:
            keyword_cloud_path = service.generate_keyword_cloud(queries, url, report_id)
        
        url_metric = URLMetrics(
            report_id=report_id,
            url=url,
            clicks=gsc.get('clicks', 0),
            impressions=gsc.get('impressions', 0),
            ctr=gsc.get('ctr', 0),
            position=gsc.get('position', 0),
            sessions=ga4.get('sessions', 0),
            pageviews=ga4.get('pageviews', 0),
            avg_session_duration=ga4.get('avg_session_duration', 0),
            bounce_rate=ga4.get('bounce_rate', 0),
            lcp=ga4.get('cwv', {}).get('lcp', 0),
            fid=ga4.get('cwv', {}).get('fid', 0),
            cls=ga4.get('cwv', {}).get('cls', 0),
            rich_results=gsc.get('rich_results', []),
            queries=queries,
            keyword_cloud_path=keyword_cloud_path,
            publish_date=datetime.fromisoformat(metadata['publish_date']).date() if metadata.get('publish_date') else None,
            change_details=metadata.get('change_details')
        )
        
        db.session.add(url_metric)
        url_metrics_list.append({
            'url': url,
            'clicks': url_metric.clicks,
            'impressions': url_metric.impressions,
            'ctr': url_metric.ctr,
            'position': url_metric.position,
            'sessions': url_metric.sessions,
            'queries': queries,
            'rich_results': url_metric.rich_results,
            'cwv': {
                'lcp': url_metric.lcp,
                'fid': url_metric.fid,
                'cls': url_metric.cls
            }
        })
    
    # Calculate totals
    totals = {
        'clicks': sum(um['clicks'] for um in url_metrics_list),
        'impressions': sum(um['impressions'] for um in url_metrics_list),
        'sessions': sum(um['sessions'] for um in url_metrics_list),
        'ctr': 0,
        'avg_position': 0
    }
    
    if totals['impressions'] > 0:
        totals['ctr'] = (totals['clicks'] / totals['impressions']) * 100
    
    if url_metrics_list:
        totals['avg_position'] = sum(um['position'] for um in url_metrics_list) / len(url_metrics_list)
    
    # Generate insights and recommendations
    insights_text, recommendations = service.generate_insights_and_recommendations(
        url_metrics_list,
        totals,
        {'start': start_date, 'end': end_date}
    )
    
    report.insights = insights_text
    report.recommendations = recommendations
    
    db.session.commit()
    
    return jsonify({
        'message': 'Data fetched successfully',
        'totals': totals,
        'url_count': len(url_metrics_list),
        'insights': insights_text,
        'recommendations': recommendations
    })

@main.route('/api/performance-reports/<int:report_id>/wordcloud/<int:url_metric_id>')
def get_wordcloud_image(report_id, url_metric_id):
    """Serve keyword cloud image"""
    url_metric = URLMetrics.query.get_or_404(url_metric_id)
    
    if url_metric.report_id != report_id:
        return jsonify({'error': 'Invalid report'}), 404
    
    if not url_metric.keyword_cloud_path or not os.path.exists(url_metric.keyword_cloud_path):
        return jsonify({'error': 'Word cloud not found'}), 404
    
    return send_file(url_metric.keyword_cloud_path, mimetype='image/png')

# --- Report Templates Routes ---

from app.report_templates import get_all_templates, get_template_by_id, get_template_categories

@main.route('/api/report-templates', methods=['GET'])
def get_report_templates():
    """Get all available report templates"""
    return jsonify({
        'templates': get_all_templates(),
        'categories': get_template_categories()
    })

@main.route('/api/report-templates/<int:template_id>', methods=['GET'])
def get_report_template(template_id):
    """Get specific template details"""
    template = get_template_by_id(template_id)
    if not template:
        return jsonify({'error': 'Template not found'}), 404
    return jsonify(template)

@main.route('/api/report-templates/<int:template_id>/generate', methods=['POST'])
def generate_from_template(template_id):
    """Generate a report based on a template"""
    template = get_template_by_id(template_id)
    if not template:
        return jsonify({'error': 'Template not found'}), 404
    
    data = request.json
    
    # Create a new performance report with template configuration
    report = PerformanceReport(
        name=f"{template['name']} - {datetime.now().strftime('%Y-%m-%d %H:%M')}",
        urls=data.get('urls', []),
        start_date=datetime.fromisoformat(data['start_date']).date() if data.get('start_date') else None,
        end_date=datetime.fromisoformat(data['end_date']).date() if data.get('end_date') else None
    )
    
    db.session.add(report)
    db.session.commit()
    
    return jsonify({
        'message': f'Report created from template: {template["name"]}',
        'report_id': report.id,
        'template_config': template['config']
    }), 201

# --- Chart and Visualization Data Routes ---

@main.route('/api/performance-reports/<int:report_id>/chart-data', methods=['GET'])
def get_chart_data(report_id):
    """
    Get chart-ready data for visualization
    Returns data formatted for Chart.js and other charting libraries
    """
    report = PerformanceReport.query.get_or_404(report_id)
    url_metrics = URLMetrics.query.filter_by(report_id=report_id).all()
    
    # Prepare data for various chart types
    chart_data = {
        'bar_chart': {
            'labels': [um.url.split('/')[-1][:30] for um in url_metrics],  # Shortened URLs
            'datasets': [
                {
                    'label': 'Clicks',
                    'data': [um.clicks for um in url_metrics],
                    'backgroundColor': 'rgba(59, 130, 246, 0.6)',
                    'borderColor': 'rgba(59, 130, 246, 1)',
                    'borderWidth': 1
                },
                {
                    'label': 'Sessions',
                    'data': [um.sessions for um in url_metrics],
                    'backgroundColor': 'rgba(16, 185, 129, 0.6)',
                    'borderColor': 'rgba(16, 185, 129, 1)',
                    'borderWidth': 1
                }
            ]
        },
        'pie_chart': {
            'labels': [um.url.split('/')[-1][:30] for um in url_metrics],
            'datasets': [{
                'data': [um.clicks for um in url_metrics],
                'backgroundColor': [
                    'rgba(59, 130, 246, 0.8)',
                    'rgba(16, 185, 129, 0.8)',
                    'rgba(251, 146, 60, 0.8)',
                    'rgba(168, 85, 247, 0.8)',
                    'rgba(236, 72, 153, 0.8)',
                    'rgba(249, 115, 22, 0.8)',
                ]
            }]
        },
        'ctr_performance': {
            'labels': [um.url.split('/')[-1][:30] for um in url_metrics],
            'datasets': [{
                'label': 'CTR %',
                'data': [um.ctr for um in url_metrics],
                'backgroundColor': 'rgba(168, 85, 247, 0.6)',
                'borderColor': 'rgba(168, 85, 247, 1)',
                'borderWidth': 2,
                'fill': True
            }]
        },
        'position_ranking': {
            'labels': [um.url.split('/')[-1][:30] for um in url_metrics],
            'datasets': [{
                'label': 'Average Position',
                'data': [um.position for um in url_metrics],
                'backgroundColor': 'rgba(236, 72, 153, 0.6)',
                'borderColor': 'rgba(236, 72, 153, 1)',
                'borderWidth': 2
            }]
        },
        'cwv_metrics': {
            'labels': ['LCP (sec)', 'FID (ms)', 'CLS (score x 100)'],
            'datasets': [{
                'label': 'Average CWV Scores',
                'data': [
                    sum(um.lcp for um in url_metrics) / len(url_metrics) if url_metrics else 0,
                    sum(um.fid for um in url_metrics) / len(url_metrics) if url_metrics else 0,
                    (sum(um.cls for um in url_metrics) / len(url_metrics) * 100) if url_metrics else 0
                ],
                'backgroundColor': [
                    'rgba(16, 185, 129, 0.6)',
                    'rgba(251, 146, 60, 0.6)',
                    'rgba(239, 68, 68, 0.6)'
                ]
            }]
        }
    }
    
    return jsonify(chart_data)

@main.route('/api/performance-reports/<int:report_id>/export', methods=['GET'])
def export_performance_report(report_id):
    """Export performance report data as CSV"""
    report = PerformanceReport.query.get_or_404(report_id)
    url_metrics = URLMetrics.query.filter_by(report_id=report_id).all()
    
    # Create CSV content
    csv_lines = [
        'URL,Clicks,Impressions,CTR %,Position,Sessions,Pageviews,Bounce Rate %,LCP,FID,CLS,Rich Results,Query Count,Publish Date,Change Details'
    ]
    
    for um in url_metrics:
        csv_lines.append(','.join([
            f'"{um.url}"',
            str(um.clicks),
            str(um.impressions),
            f'{um.ctr:.2f}',
            f'{um.position:.1f}',
            str(um.sessions),
            str(um.pageviews),
            f'{um.bounce_rate:.2f}',
            f'{um.lcp:.2f}',
            f'{um.fid:.2f}',
            f'{um.cls:.3f}',
            f'"{",".join(um.rich_results) if um.rich_results else ""}"',
            str(len(um.queries) if um.queries else 0),
            um.publish_date.isoformat() if um.publish_date else '',
            f'"{um.change_details or ""}"'
        ]))
    
    csv_content = '\n'.join(csv_lines)
    
    # Create response
    from io import BytesIO
    output = BytesIO()
    output.write(csv_content.encode('utf-8'))
    output.seek(0)
    
    return send_file(
        output,
        mimetype='text/csv',
        as_attachment=True,
        download_name=f'performance_report_{report_id}_{datetime.now().strftime("%Y%m%d")}.csv'
    )

