import * as React from 'react';
import { AxisDomain, BaseAxisProps, ScaleType } from '../util/types';
import { AxisId } from '../state/cartesianAxisSlice';
import { AxisRange } from '../state/selectors/axisSelectors';
import { CustomScaleDefinition } from '../util/scale/CustomScaleDefinition';
export interface Props extends Omit<BaseAxisProps, 'domain'> {
    /**
     * The type of axis.
     *
     * `category`: Treats data as distinct values.
     * Each value is in the same distance from its neighbors, regardless of their actual numeric difference.
     *
     * `number`: Treats data as continuous range.
     * Values that are numerically closer are placed closer together on the axis.
     *
     * @defaultValue number
     */
    type?: 'number' | 'category';
    /**
     * The unique id of z-axis.
     *
     * @defaultValue 0
     */
    zAxisId?: AxisId;
    /**
     * The range of axis.
     * Unlike other axes, the range of z-axis is not informed by chart dimensions.
     *
     * @defaultValue [64,64]
     */
    range?: AxisRange;
    /**
     * Specify the domain of axis when the axis is a number axis.
     *
     * If undefined, then the domain is calculated based on the data and dataKeys.
     *
     * The length of domain should be 2, and we will validate the values in domain.
     *
     * Each element in the array can be a number, 'auto', 'dataMin', 'dataMax', a string like 'dataMin - 20', 'dataMax + 100',
     * or a function that accepts a single argument and returns a number.
     *
     * If any element of domain is set to be 'auto', comprehensible scale ticks will be calculated, and the final domain of axis is generated by the ticks.
     * If a function, receives '[dataMin, dataMax]', and must return a computed domain as '[min, max]'.
     *
     * @example <ZAxis type="number" domain={['dataMin', 'dataMax']} />
     * @example <ZAxis type="number" domain={[0, 'dataMax']} />
     * @example <ZAxis type="number" domain={['auto', 'auto']} />
     * @example <ZAxis type="number" domain={[0, 'dataMax + 1000']} />
     * @example <ZAxis type="number" domain={['dataMin - 100', 'dataMax + 100']} />
     * @example <ZAxis type="number" domain={[dataMin => (0 - Math.abs(dataMin)), dataMax => (dataMax * 2)]} />
     * @example <ZAxis type="number" domain={([dataMin, dataMax]) => { const absMax = Math.max(Math.abs(dataMin), Math.abs(dataMax)); return [-absMax, absMax]; }} />
     * @example <ZAxis type="number" domain={[0, 100]} allowDataOverflow />
     */
    domain?: AxisDomain;
    /**
     * Scale function determines how data values are mapped to visual values.
     * In other words, decided the mapping between data domain and coordinate range.
     *
     * If undefined, or 'auto', the scale function is created internally according to the type of axis and data.
     *
     * You can define a custom scale, either as a string shortcut to a d3 scale, or as a complete scale definition object.
     *
     * @defaultValue auto
     * @example <ZAxis scale="log" />
     * @example
     * import { scaleLog } from 'd3-scale';
     * const scale = scaleLog().base(Math.E);
     * <ZAxis scale={scale} />
     */
    scale?: ScaleType | CustomScaleDefinition | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
}
export declare const zAxisDefaultProps: {
    readonly zAxisId: 0;
    readonly range: AxisRange;
    readonly scale: ScaleType | CustomScaleDefinition<import("../util/types").CategoricalDomainItem> | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    readonly type: import("../util/types").EvaluatedAxisDomainType;
};
/**
 * Virtual axis, does not render anything itself. Has no ticks, grid lines, or labels.
 * Useful for dynamically setting Scatter point size, based on data.
 *
 * @consumes CartesianViewBoxContext
 */
export declare function ZAxis(outsideProps: Props): React.JSX.Element;
export declare namespace ZAxis {
    var displayName: string;
}
