import * as React from 'react';
import { ReactElement, ReactNode, SVGProps } from 'react';
import { ActiveShape, AnimationDuration, AnimationTiming, ChartOffsetInternal, Coordinate, DataKey, DataProvider, GeometrySector, LegendType, PresentationAttributesAdaptChildEvent, TooltipType } from '../util/types';
import { TooltipPayload } from '../state/tooltipSlice';
import { PiePresentationProps, PieSettings } from '../state/types/PieSettings';
import { SVGPropsNoEvents } from '../util/svgPropertiesNoEvents';
import { Props as LabelListProps } from '../component/LabelList';
import { ZIndexable } from '../zIndex/ZIndexLayer';
import { ChartData } from '../state/chartDataSlice';
interface PieDef {
    /**
     * The x-coordinate of center. If set a percentage, the final value is obtained by multiplying the percentage of container width.
     */
    cx?: number | string;
    /**
     * The y-coordinate of center. If set a percentage, the final value is obtained by multiplying the percentage of container height.
     */
    cy?: number | string;
    /**
     * Angle in degrees from which the chart should start.
     */
    startAngle?: number;
    /**
     * Angle, in degrees, at which the chart should end.
     */
    endAngle?: number;
    /**
     * The angle between two sectors.
     *
     * @example <Pie paddingAngle={5} />
     * @example https://recharts.github.io/examples/PieChartWithPaddingAngle
     */
    paddingAngle?: number;
    /**
     * The inner radius of the sectors.
     * If set a percentage, the final value is obtained by multiplying the percentage of maxRadius which is calculated by the width, height, cx, cy.
     */
    innerRadius?: number | string;
    /**
     * The outer radius of the sectors.
     * If set a percentage, the final value is obtained by multiplying the percentage of maxRadius which is calculated by the width, height, cx, cy.
     * Function should return a string percentage or number.
     */
    outerRadius?: number | string | ((dataPoint: any) => number | string);
    cornerRadius?: number | string;
}
type PieLabelLine = ReactElement<SVGElement> | ((props: any) => ReactElement<SVGElement>) | SVGProps<SVGPathElement> | boolean;
interface PieLabelExtraProps {
    stroke: string;
    index: number;
    textAnchor: string;
}
export type PieLabelRenderProps = Omit<SVGPropsNoEvents<PieSvgAttributes>, 'offset'> & Omit<PieSectorDataItem, 'offset'> & PieLabelExtraProps & Coordinate;
export type LabelListPropsWithPosition = LabelListProps & {
    position: LabelListProps['position'];
};
/**
 * The `label` prop in Pie accepts a variety of alternatives.
 */
export type PieLabel = boolean | LabelListPropsWithPosition | Partial<PieLabelRenderProps> | ((props: PieLabelRenderProps) => ReactNode | ReactElement<SVGElement>) | ReactElement<SVGElement>;
export type PieSectorData = GeometrySector & {
    dataKey?: DataKey<any>;
    midAngle?: number;
    middleRadius?: number;
    name?: string | number;
    paddingAngle?: number;
    payload?: any;
    percent?: number;
    tooltipPayload?: TooltipPayload;
    tooltipPosition: Coordinate;
    value: number;
};
/**
 * We spread the data object into the sector data item,
 * so we can't really know what is going to be inside.
 *
 * This type represents our best effort, but it all depends on the input data
 * and what is inside of it.
 *
 * https://github.com/recharts/recharts/issues/6380
 * https://github.com/recharts/recharts/discussions/6375
 */
export type PieSectorDataItem = PiePresentationProps & PieCoordinate & PieSectorData & {
    cornerRadius: number | undefined;
};
export type PieSectorShapeProps = PieSectorDataItem & {
    isActive: boolean;
    index: number;
};
export type PieShape = ReactNode | ((props: PieSectorShapeProps, index: number) => React.ReactElement);
interface PieProps extends DataProvider, PieDef, ZIndexable {
    /**
     * This component is rendered when this graphical item is activated
     * (could be by mouse hover, touch, keyboard, programmatically).
     *
     * @deprecated Use the `shape` prop to create each sector. `isActive` designates the "active" shape.
     * @example <Pie activeShape={<CustomActiveShape />} />
     * @example https://recharts.github.io/examples/CustomActiveShapePieChart
     */
    activeShape?: ActiveShape<PieSectorDataItem>;
    /**
     * Specifies when the animation should begin, the unit of this option is ms.
     * @defaultValue 400
     */
    animationBegin?: number;
    /**
     * Specifies the duration of animation, the unit of this option is ms.
     * @defaultValue 1500
     */
    animationDuration?: AnimationDuration;
    /**
     * The type of easing function.
     * @defaultValue ease
     */
    animationEasing?: AnimationTiming;
    className?: string;
    /**
     * Decides how to extract the value of this Pie from the data:
     * - `string`: the name of the field in the data object;
     * - `number`: the index of the field in the data;
     * - `function`: a function that receives the data object and returns the value of this Pie.
     *
     * @defaultValue value
     */
    dataKey?: DataKey<any>;
    /**
     * Hides the whole graphical element when true.
     *
     * Hiding an element is different from removing it from the chart:
     * Hidden graphical elements are still visible in Legend,
     * and can be included in axis domain calculations,
     * depending on `includeHidden` props of your XAxis/YAxis.
     *
     * @defaultValue false
     */
    hide?: boolean;
    id?: string;
    /**
     * The shape of inactive sector.
     * @deprecated Use the `shape` prop to modify each sector.
     */
    inactiveShape?: ActiveShape<PieSectorDataItem>;
    /**
     * If set false, animation will be disabled.
     * If set "auto", the animation will be disabled in SSR and enabled in browser.
     * @defaultValue auto
     */
    isAnimationActive?: boolean | 'auto';
    /**
     * Renders one label for each pie sector. Options:
     * - `true`: renders default labels;
     * - `false`: no labels are rendered;
     * - `object` that has `position` prop: the props of LabelList component;
     * - `object` that does not have `position` prop: the props of a custom Pie label (similar to Label with position "outside"); this variant supports `labelLine`
     * - `ReactElement`: a custom label element;
     * - `function`: a render function of custom label.
     *
     * Also see the `labelLine` prop that draws a line connecting each label to the corresponding sector.
     *
     * @defaultValue false
     * @example <Pie label={<CustomizedLabel />} />
     * @example https://recharts.github.io/examples/PieChartWithCustomizedLabel
     */
    label?: PieLabel;
    /**
     * If false set, label lines will not be drawn. If true set, label lines will be drawn which have the props calculated internally.
     * If object set, label lines will be drawn which have the props merged by the internal calculated props and the option.
     * If ReactElement set, the option can be the custom label line element.
     * If set a function, the function will be called to render customized label line.
     * @defaultValue true
     * @example <Pie labelLine={<CustomizedLabelLine />} />
     * @example https://recharts.github.io/examples/PieChartWithCustomizedLabel
     */
    labelLine?: PieLabelLine;
    /**
     * The type of icon in legend. If set to 'none', no legend item will be rendered.
     * @defaultValue rect
     */
    legendType?: LegendType;
    /** the max radius of pie */
    maxRadius?: number;
    /**
     * The minimum angle of each unzero data.
     * @defaultValue 0
     */
    minAngle?: number;
    /**
     * Name represents each sector in the tooltip, and legend.
     * This allows you to extract the name from the data:
     *
     * - `string`: the name of the field in the data object;
     * - `number`: the index of the field in the data;
     * - `function`: a function that receives the data object and returns the name.
     *
     * @defaultValue name
     */
    nameKey?: DataKey<any>;
    /**
     * The customized event handler of animation end.
     */
    onAnimationEnd?: () => void;
    /**
     * The customized event handler of animation start.
     */
    onAnimationStart?: () => void;
    /**
     * The customized event handler of click on the sectors in this group.
     */
    onClick?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mousedown on the sectors in this group.
     */
    onMouseDown?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseup on the sectors in this group.
     */
    onMouseUp?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mousemove on the sectors in this group.
     */
    onMouseMove?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseover on the sectors in this group.
     */
    onMouseOver?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseout on the sectors in this group.
     */
    onMouseOut?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseenter on the sectors in this group.
     */
    onMouseEnter?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseleave on the sectors in this group.
     */
    onMouseLeave?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The tabindex of wrapper surrounding the cells.
     * @defaultValue 0
     */
    rootTabIndex?: number;
    /**
     * The custom shape of a Pie Sector.
     * Can also be used to render active sector by checking isActive.
     * If undefined, renders {@link Sector} shape.
     */
    shape?: PieShape;
    tooltipType?: TooltipType;
    /**
     * @defaultValue 100
     */
    zIndex?: number;
}
type PieSvgAttributes = Omit<PresentationAttributesAdaptChildEvent<any, SVGElement>, 'ref'>;
export type Props = PieSvgAttributes & PieProps;
export type PieCoordinate = {
    cx: number;
    cy: number;
    innerRadius: number;
    outerRadius: number;
    maxRadius: number;
};
export declare function computePieSectors({ pieSettings, displayedData, cells, offset, }: {
    pieSettings: PieSettings;
    displayedData: ChartData;
    cells: ReadonlyArray<ReactElement> | undefined;
    offset: ChartOffsetInternal;
}): ReadonlyArray<PieSectorDataItem> | undefined;
export declare const defaultPieProps: {
    readonly animationBegin: 400;
    readonly animationDuration: 1500;
    readonly animationEasing: "ease";
    readonly cx: "50%";
    readonly cy: "50%";
    readonly dataKey: "value";
    readonly endAngle: 360;
    readonly fill: "#808080";
    readonly hide: false;
    readonly innerRadius: 0;
    readonly isAnimationActive: "auto";
    readonly label: false;
    readonly labelLine: true;
    readonly legendType: "rect";
    readonly minAngle: 0;
    readonly nameKey: "name";
    readonly outerRadius: "80%";
    readonly paddingAngle: 0;
    readonly rootTabIndex: 0;
    readonly startAngle: 0;
    readonly stroke: "#fff";
    readonly zIndex: 100;
};
/**
 * @consumes PolarChartContext
 * @provides LabelListContext
 * @provides CellReader
 */
export declare function Pie(outsideProps: Props): React.JSX.Element;
export declare namespace Pie {
    var displayName: string;
}
export {};
