import React, { useState, useEffect } from 'react';
import Modal from 'react-bootstrap/Modal';
import Button from 'react-bootstrap/Button';
import Row from 'react-bootstrap/Row';
import Col from 'react-bootstrap/Col';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import { docsAPI } from '../services/api';
import { useToast } from '../context/ToastContext';

interface DocsOverlayProps {
    show: boolean;
    onHide: () => void;
}

const DocsOverlay: React.FC<DocsOverlayProps> = ({ show, onHide }) => {
    const [files, setFiles] = useState<string[]>([]);
    const [selectedFile, setSelectedFile] = useState<string | null>(null);
    const [content, setContent] = useState<string>('');
    const [loading, setLoading] = useState(false);
    const { showToast } = useToast();

    useEffect(() => {
        if (show) {
            loadFileList();
        }
    }, [show]);

    useEffect(() => {
        if (selectedFile) {
            loadFileContent(selectedFile);
        }
    }, [selectedFile]);

    const loadFileList = async () => {
        try {
            const response = await docsAPI.list();
            setFiles(response.data);
            if (response.data.length > 0 && !selectedFile) {
                setSelectedFile(response.data[0]);
            }
        } catch (error) {
            console.error('Failed to load doc list:', error);
            showToast('Failed to load documentation list', 'error');
        }
    };

    const loadFileContent = async (filename: string) => {
        setLoading(true);
        try {
            const response = await docsAPI.get(filename);
            setContent(response.data.content);
        } catch (error) {
            console.error('Failed to load doc content:', error);
            showToast(`Failed to load ${filename}`, 'error');
        } finally {
            setLoading(false);
        }
    };

    return (
        <Modal show={show} onHide={onHide} size="xl" fullscreen className="docs-modal">
            <Modal.Header closeButton style={{ background: 'var(--bg-card)', borderBottom: '1px solid var(--border)' }}>
                <Modal.Title style={{ color: 'var(--text-main)' }}>
                    📚 Project Documentation
                </Modal.Title>
            </Modal.Header>
            <Modal.Body style={{ background: 'var(--bg-main)', padding: 0 }}>
                <Row className="h-100 g-0">
                    {/* Sidebar Navigation */}
                    <Col md={3} lg={2} style={{
                        background: 'var(--bg-card)',
                        borderRight: '1px solid var(--border)',
                        height: 'calc(100vh - 60px)',
                        overflowY: 'auto'
                    }}>
                        <div className="p-3">
                            <h6 className="text-muted text-uppercase small fw-bold mb-3">Guides & References</h6>
                            <div className="d-grid gap-2">
                                {files.map(file => (
                                    <Button
                                        key={file}
                                        variant={selectedFile === file ? "primary" : "outline-secondary"}
                                        size="sm"
                                        className="text-start text-truncate"
                                        onClick={() => setSelectedFile(file)}
                                        style={{
                                            border: 'none',
                                            background: selectedFile === file ? 'var(--primary)' : 'transparent',
                                            color: selectedFile === file ? 'white' : 'var(--text-main)',
                                            opacity: selectedFile === file ? 1 : 0.7
                                        }}
                                    >
                                        {file.replace(/_/g, ' ').replace('.md', '')}
                                    </Button>
                                ))}
                            </div>
                        </div>
                    </Col>

                    {/* Content Area */}
                    <Col md={9} lg={10} style={{
                        height: 'calc(100vh - 60px)',
                        overflowY: 'auto',
                        padding: '2rem'
                    }}>
                        <div style={{ maxWidth: '900px', margin: '0 auto', color: 'var(--text-main)' }}>
                            {loading ? (
                                <div className="text-center p-5 text-muted">Loading...</div>
                            ) : (
                                <div className="markdown-content">
                                    <h1 className="mb-4 pb-2 border-bottom">{selectedFile?.replace(/_/g, ' ').replace('.md', '')}</h1>
                                    <ReactMarkdown remarkPlugins={[remarkGfm]}>
                                        {content}
                                    </ReactMarkdown>
                                </div>
                            )}
                        </div>
                    </Col>
                </Row>
            </Modal.Body>
        </Modal>
    );
};

export default DocsOverlay;
