import React, { useState } from 'react';
import { Link, useLocation } from 'react-router-dom';
import Container from 'react-bootstrap/Container';
import Nav from 'react-bootstrap/Nav';
import Navbar from 'react-bootstrap/Navbar';
import Dropdown from 'react-bootstrap/Dropdown';
import Form from 'react-bootstrap/Form';
import { useTheme } from '../../context/ThemeContext';

import DocsOverlay from '../DocsOverlay';

interface LayoutProps {
    children: React.ReactNode;
}

const Layout: React.FC<LayoutProps> = ({ children }) => {
    const location = useLocation();
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [showDocs, setShowDocs] = useState(false);

    const navItems = [
        { path: '/', label: 'Dashboard', icon: '📊' },
        { path: '/strategy/new', label: 'New Strategy', icon: '➕' },
        { path: '/reports', label: 'Reports', icon: '📈' },
        { path: '/quotes', label: 'TEST', icon: '💬' },
    ];

    const { theme, setTheme } = useTheme();

    return (
        <div className="d-flex">
            <DocsOverlay show={showDocs} onHide={() => setShowDocs(false)} />

            {/* Glassy Floating Help Button */}
            <div
                className="btn-glassy-floating"
                onClick={() => setShowDocs(true)}
                title="Documentation"
            >
                ?
            </div>

            {/* Sidebar */}
            <div className={`sidebar ${sidebarOpen ? 'show' : ''} d-flex flex-column`} style={{ width: '250px' }}>
                <div className="p-4">
                    <h4 className="mb-0" style={{ color: 'var(--primary)' }}>
                        SEO Tracker V3
                    </h4>
                    <small style={{ color: 'var(--text-muted)' }}>Strategy Management</small>
                </div>

                <Nav className="flex-column px-3">
                    {navItems.map((item) => (
                        <Nav.Link
                            key={item.path}
                            as={Link}
                            to={item.path}
                            className={`nav-link-custom mb-2 ${location.pathname === item.path ? 'active' : ''}`}
                        >
                            <span className="me-2">{item.icon}</span>
                            {item.label}
                        </Nav.Link>
                    ))}
                </Nav>

                {/* Theme Toggle */}
                <div className="px-4 py-2 mt-auto">
                    <div className="d-flex align-items-center justify-content-between p-2 rounded" style={{ background: 'var(--bg-hover)' }}>
                        <span style={{ fontSize: '1.2rem' }}>☀️</span>
                        <Form.Check
                            type="switch"
                            id="theme-switch"
                            checked={theme === 'dark'}
                            onChange={() => setTheme(theme === 'dark' ? 'light' : 'dark')}
                            style={{ transform: 'scale(1.2)' }}
                        />
                        <span style={{ fontSize: '1.2rem' }}>🌙</span>
                    </div>
                </div>

                {/* User Profile */}
                <div className="w-100 p-3" style={{ borderTop: '1px solid var(--border)' }}>
                    <Dropdown drop="up">
                        <Dropdown.Toggle variant="link" className="w-100 text-decoration-none text-start" style={{ color: 'var(--text-main)' }}>
                            <div className="d-flex align-items-center">
                                <div className="rounded-circle bg-primary me-2" style={{ width: '32px', height: '32px' }}></div>
                                <div>
                                    <div className="small fw-bold">Admin User</div>
                                    <div className="text-muted" style={{ fontSize: '0.75rem' }}>admin@jetlife.com</div>
                                </div>
                            </div>
                        </Dropdown.Toggle>

                        <Dropdown.Menu>
                            <Dropdown.Item onClick={() => setShowDocs(true)}>Documentation</Dropdown.Item>
                            <Dropdown.Item as={Link} to="/settings">Settings</Dropdown.Item>
                            <Dropdown.Divider />
                            <Dropdown.Item>Sign Out</Dropdown.Item>
                        </Dropdown.Menu>
                    </Dropdown>
                </div>
            </div>

            {/* Main Content */}
            <div className="flex-grow-1" style={{ minHeight: '100vh', background: 'var(--bg-main)' }}>
                {/* Top Navbar (Mobile) */}
                <Navbar expand="lg" className="d-lg-none" style={{ background: 'var(--bg-card)', borderBottom: '1px solid var(--border)' }}>
                    <Container fluid>
                        <Navbar.Brand style={{ color: 'var(--primary)' }}>SEO Tracker V3</Navbar.Brand>
                        <Navbar.Toggle aria-controls="mobile-nav" onClick={() => setSidebarOpen(!sidebarOpen)} />
                    </Container>
                </Navbar>

                {/* Page Content */}
                <Container fluid className="p-4">
                    {children}
                </Container>
            </div>
        </div>
    );
};

export default Layout;
