import React, { createContext, useContext, useState, useEffect } from 'react';
import type { ReactNode } from 'react';
import { settingsAPI } from '../services/api';

interface ThemeContextType {
    theme: string;
    setTheme: (theme: string) => void;
}

const ThemeContext = createContext<ThemeContextType | undefined>(undefined);

export const ThemeProvider: React.FC<{ children: ReactNode }> = ({ children }) => {
    const [theme, setThemeState] = useState<string>('dark');

    useEffect(() => {
        // Load theme from settings
        settingsAPI.get()
            .then(res => {
                // Assuming settings backend returns 'theme_variant' as string now or we map it
                // For now, let's just stick to local state or default dirty check, 
                // but actually we should map: if it was a number before, default to 'dark'
                if (res.data.theme_variant && (res.data.theme_variant === 'light' || res.data.theme_variant === 'dark')) {
                    setThemeState(res.data.theme_variant);
                }
            })
            .catch(err => console.error('Failed to load theme:', err));
    }, []);

    useEffect(() => {
        // Apply theme class to body
        document.body.className = `theme-${theme}`;
    }, [theme]);

    const setTheme = (newTheme: string) => {
        setThemeState(newTheme);
        // Save to backend
        settingsAPI.update({ theme_variant: newTheme })
            .catch(err => console.error('Failed to save theme:', err));
    };

    return (
        <ThemeContext.Provider value={{ theme, setTheme }}>
            {children}
        </ThemeContext.Provider>
    );
};

export const useTheme = () => {
    const context = useContext(ThemeContext);
    if (!context) {
        throw new Error('useTheme must be used within ThemeProvider');
    }
    return context;
};
