import React, { createContext, useContext, useState, useCallback } from 'react';
import type { ReactNode } from 'react';
import Toast from 'react-bootstrap/Toast';
import ToastContainer from 'react-bootstrap/ToastContainer';

type ToastType = 'success' | 'error' | 'info' | 'warning';

interface ToastMessage {
    id: number;
    message: string;
    type: ToastType;
}

interface ToastContextType {
    showToast: (message: string, type?: ToastType) => void;
}

const ToastContext = createContext<ToastContextType | undefined>(undefined);

export const ToastProvider: React.FC<{ children: ReactNode }> = ({ children }) => {
    const [toasts, setToasts] = useState<ToastMessage[]>([]);

    const showToast = useCallback((message: string, type: ToastType = 'info') => {
        const id = Date.now();
        setToasts((prev) => [...prev, { id, message, type }]);

        // Auto-remove after 5 seconds (optional as Bootstrap Toast has autohide)
        setTimeout(() => {
            setToasts((prev) => prev.filter((t) => t.id !== id));
        }, 5000);
    }, []);

    const removeToast = (id: number) => {
        setToasts((prev) => prev.filter((t) => t.id !== id));
    };

    return (
        <ToastContext.Provider value={{ showToast }}>
            {children}
            <ToastContainer position="top-end" className="p-3" style={{ zIndex: 9999 }}>
                {toasts.map((toast) => (
                    <Toast
                        key={toast.id}
                        onClose={() => removeToast(toast.id)}
                        show={true}
                        delay={5000}
                        autohide
                        className={`toast-glassy toast-${toast.type}`}
                        style={{
                            background: 'rgba(255, 255, 255, 0.15)',
                            backdropFilter: 'blur(12px)',
                            border: '1px solid rgba(255, 255, 255, 0.3)',
                            boxShadow: '0 8px 32px 0 rgba(31, 38, 135, 0.37)',
                            borderRadius: '12px',
                            color: 'white',
                            marginBottom: '1rem',
                            transform: 'perspective(1000px) rotateX(10deg)',
                            transition: 'all 0.3s ease',
                            overflow: 'hidden'
                        }}
                    >
                        <Toast.Header
                            style={{
                                background: 'rgba(255, 255, 255, 0.1)',
                                borderBottom: '1px solid rgba(255, 255, 255, 0.2)',
                                color: 'white'
                            }}
                        >
                            <strong className="me-auto" style={{ textTransform: 'uppercase', letterSpacing: '1px' }}>
                                {toast.type === 'success' && '✅ Success'}
                                {toast.type === 'error' && '❌ Error'}
                                {toast.type === 'warning' && '⚠️ Warning'}
                                {toast.type === 'info' && 'ℹ️ Info'}
                            </strong>
                            <small>Just now</small>
                        </Toast.Header>
                        <Toast.Body style={{ fontSize: '1.05rem', fontWeight: 500 }}>
                            {toast.message}
                        </Toast.Body>
                        <div
                            style={{
                                position: 'absolute',
                                top: 0,
                                left: 0,
                                right: 0,
                                bottom: 0,
                                background: 'linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0) 100%)',
                                pointerEvents: 'none'
                            }}
                        />
                    </Toast>
                ))}
            </ToastContainer>
        </ToastContext.Provider>
    );
};

export const useToast = () => {
    const context = useContext(ToastContext);
    if (!context) {
        throw new Error('useToast must be used within a ToastProvider');
    }
    return context;
};
