import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import Container from 'react-bootstrap/Container';
import Row from 'react-bootstrap/Row';
import Col from 'react-bootstrap/Col';
import Card from 'react-bootstrap/Card';
import Button from 'react-bootstrap/Button';
import Badge from 'react-bootstrap/Badge';
import { strategyAPI } from '../services/api';

interface Strategy {
    id: number;
    name: string;
    description: string;
    status: string;
    created_at: string;
    benchmark_data_count: number;
    performance_data_count: number;
}

const Dashboard: React.FC = () => {
    const [strategies, setStrategies] = useState<Strategy[]>([]);
    const [loading, setLoading] = useState(true);
    const [filter, setFilter] = useState<string>('');

    useEffect(() => {
        loadStrategies();
    }, [filter]);

    const loadStrategies = async () => {
        try {
            const response = await strategyAPI.list(filter || undefined);
            setStrategies(response.data);
        } catch (error) {
            console.error('Failed to load strategies:', error);
        } finally {
            setLoading(false);
        }
    };

    const getStatusBadgeVariant = (status: string) => {
        const statusLower = status.toLowerCase();
        if (statusLower.includes('live')) return 'success';
        if (statusLower.includes('pending')) return 'primary';
        return 'warning';
    };

    if (loading) {
        return <div>Loading...</div>;
    }

    return (
        <Container fluid>
            <Row className="mb-4">
                <Col>
                    <h1 style={{ color: 'var(--text-main)' }}>Dashboard</h1>
                    <p style={{ color: 'var(--text-muted)' }}>Manage your SEO strategies and track performance</p>
                </Col>
                <Col xs="auto">
                    <Link to="/strategy/new">
                        <Button variant="primary" size="lg" className="btn-primary-custom">
                            + New Strategy
                        </Button>
                    </Link>
                </Col>
            </Row>

            {/* Filter Buttons */}
            <Row className="mb-4">
                <Col>
                    <div className="d-flex gap-2">
                        <Button
                            variant={filter === '' ? 'primary' : 'outline-secondary'}
                            onClick={() => setFilter('')}
                        >
                            All
                        </Button>
                        <Button
                            variant={filter === 'Planning' ? 'primary' : 'outline-secondary'}
                            onClick={() => setFilter('Planning')}
                        >
                            Planning
                        </Button>
                        <Button
                            variant={filter === 'Pending' ? 'primary' : 'outline-secondary'}
                            onClick={() => setFilter('Pending')}
                        >
                            Pending
                        </Button>
                        <Button
                            variant={filter === 'Live - Collecting Live Test Data' ? 'primary' : 'outline-secondary'}
                            onClick={() => setFilter('Live - Collecting Live Test Data')}
                        >
                            Live
                        </Button>
                    </div>
                </Col>
            </Row>

            {/* Strategy Cards */}
            <Row>
                {strategies.length === 0 ? (
                    <Col>
                        <Card className="card-custom text-center p-5">
                            <h3 style={{ color: 'var(--text-muted)' }}>No strategies found</h3>
                            <p style={{ color: 'var(--text-muted)' }}>Create your first strategy to get started!</p>
                            <Link to="/strategy/new">
                                <Button variant="primary" className="btn-primary-custom">
                                    Create Strategy
                                </Button>
                            </Link>
                        </Card>
                    </Col>
                ) : (
                    strategies.map((strategy) => (
                        <Col key={strategy.id} md={6} lg={4} className="mb-4">
                            <Card className="card-custom h-100">
                                <Card.Body>
                                    <div className="d-flex justify-content-between align-items-start mb-3">
                                        <Card.Title style={{ color: 'var(--text-main)' }}>
                                            {strategy.name}
                                        </Card.Title>
                                        <Badge bg={getStatusBadgeVariant(strategy.status)}>
                                            {strategy.status}
                                        </Badge>
                                    </div>
                                    <Card.Text style={{ color: 'var(--text-muted)' }}>
                                        {strategy.description || 'No description'}
                                    </Card.Text>
                                    <div className="d-flex justify-content-between text-muted small mb-3">
                                        <span>Benchmark: {strategy.benchmark_data_count} rows</span>
                                        <span>Live: {strategy.performance_data_count} rows</span>
                                    </div>
                                    <Link to={`/strategy/${strategy.id}`}>
                                        <Button variant="outline-primary" className="w-100">
                                            View Details
                                        </Button>
                                    </Link>
                                </Card.Body>
                            </Card>
                        </Col>
                    ))
                )}
            </Row>
        </Container>
    );
};

export default Dashboard;
