import React, { useState, useEffect } from 'react';
import Container from 'react-bootstrap/Container';
import Card from 'react-bootstrap/Card';
import Form from 'react-bootstrap/Form';
import Button from 'react-bootstrap/Button';
import { useToast } from '../context/ToastContext';
import { settingsAPI } from '../services/api';

const Settings: React.FC = () => {
    const { showToast } = useToast();
    const [settings, setSettings] = useState({
        required_fields_enabled: true,
        advanced_reporting_enabled: false,
        email_notifications: true,
        update_notifications: true
    });
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        loadSettings();
    }, []);

    const loadSettings = async () => {
        try {
            const response = await settingsAPI.get();
            setSettings(response.data);
        } catch (error) {
            console.error('Failed to load settings:', error);
            showToast('Failed to load settings. Please refresh the page.', 'error');
        } finally {
            setLoading(false);
        }
    };

    const handleSave = async () => {
        try {
            await settingsAPI.update(settings);
            showToast('Settings saved successfully!', 'success');
        } catch (error) {
            console.error('Failed to save settings:', error);
            showToast('Failed to save settings. Please try again.', 'error');
        }
    };


    // Theme selection is now handled in the Sidebar layout
    if (loading) {
        return <div>Loading...</div>;
    }

    return (
        <Container fluid>
            <h1 className="mb-4" style={{ color: 'var(--text-main)' }}>Settings</h1>


            {/* Validation Settings */}
            <Card className="card-custom mb-4">
                <Card.Header>
                    <h5 className="mb-0">Validation</h5>
                </Card.Header>
                <Card.Body>
                    <Form.Check
                        type="switch"
                        id="required-fields-switch"
                        label="Enforce required fields when submitting strategies"
                        checked={settings.required_fields_enabled}
                        onChange={(e) => setSettings({ ...settings, required_fields_enabled: e.target.checked })}
                    />
                    <Form.Text className="text-muted">
                        When disabled, you can save incomplete strategies as drafts
                    </Form.Text>
                </Card.Body>
            </Card>

            {/* Advanced Reporting */}
            <Card className="card-custom mb-4">
                <Card.Header>
                    <h5 className="mb-0">Advanced Features</h5>
                </Card.Header>
                <Card.Body>
                    <Form.Check
                        type="switch"
                        id="advanced-reporting-switch"
                        label="Enable Advanced Reporting Module"
                        checked={settings.advanced_reporting_enabled}
                        onChange={(e) => setSettings({ ...settings, advanced_reporting_enabled: e.target.checked })}
                    />
                    <Form.Text className="text-muted">
                        Unlock custom report builder, scheduled reports, and multi-format exports
                    </Form.Text>
                </Card.Body>
            </Card>

            {/* Notifications */}
            <Card className="card-custom mb-4">
                <Card.Header>
                    <h5 className="mb-0">Notifications</h5>
                </Card.Header>
                <Card.Body>
                    <Form.Check
                        type="switch"
                        id="email-notifications-switch"
                        label="Email notifications"
                        checked={settings.email_notifications}
                        onChange={(e) => setSettings({ ...settings, email_notifications: e.target.checked })}
                        className="mb-3"
                    />
                    <Form.Check
                        type="switch"
                        id="update-notifications-switch"
                        label="Data update notifications"
                        checked={settings.update_notifications}
                        onChange={(e) => setSettings({ ...settings, update_notifications: e.target.checked })}
                    />
                </Card.Body>
            </Card>

            <Button variant="primary" size="lg" className="btn-primary-custom" onClick={handleSave}>
                Save Settings
            </Button>
        </Container>
    );
};

export default Settings;
