import React, { useEffect, useState } from 'react';
import Container from 'react-bootstrap/Container';
import Row from 'react-bootstrap/Row';
import Col from 'react-bootstrap/Col';
import Card from 'react-bootstrap/Card';
import Badge from 'react-bootstrap/Badge';
import Button from 'react-bootstrap/Button';
import Form from 'react-bootstrap/Form';
import ListGroup from 'react-bootstrap/ListGroup';
import Spinner from 'react-bootstrap/Spinner';
import { useParams } from 'react-router-dom';
import { strategyAPI, oauthAPI } from '../services/api';
import { useToast } from '../context/ToastContext';

// Types (should eventually be moved to types.ts)
interface Strategy {
    id: number;
    name: string;
    description: string;
    owner_name: string;
    status: string;
    go_live_date: string;
    test_end_date: string;
    benchmark_start_date: string;
    benchmark_end_date: string;
    integrations: Integration[];
    notes?: Note[];
}

interface Integration {
    id: number;
    platform: string;
    property_id: string;
    property_name: string;
    is_connected: boolean;
    connected_at: string;
}

interface Note {
    id: number;
    content: string;
    created_at: string;
}

const StrategyDetail: React.FC = () => {
    const { id } = useParams<{ id: string }>();
    const { showToast } = useToast();
    const [strategy, setStrategy] = useState<Strategy | null>(null);
    const [notes, setNotes] = useState<Note[]>([]);
    const [loading, setLoading] = useState(true);

    // Notes state
    const [newNote, setNewNote] = useState('');
    const [submittingNote, setSubmittingNote] = useState(false);

    useEffect(() => {
        fetchData();
    }, [id]);

    const fetchData = async () => {
        try {
            if (!id) return;
            const sId = parseInt(id);
            const [stratRes, notesRes] = await Promise.all([
                strategyAPI.get(sId),
                strategyAPI.getNotes(sId)
            ]);
            setStrategy(stratRes.data);
            setNotes(notesRes.data);
        } catch (error) {
            console.error(error);
            showToast('Failed to load strategy details', 'error');
        } finally {
            setLoading(false);
        }
    };

    const handleAddNote = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!newNote.trim() || !id) return;

        setSubmittingNote(true);
        try {
            const res = await strategyAPI.addNote(parseInt(id), newNote);
            setNotes([res.data, ...notes]);
            setNewNote('');
            showToast('Note added successfully', 'success');
        } catch (error) {
            console.error(error);
            showToast('Failed to add note', 'error');
        } finally {
            setSubmittingNote(false);
        }
    };

    const handleConnect = async (platform: 'GSC' | 'GA4') => {
        if (!id) return;
        try {
            let res;
            const sId = parseInt(id);
            if (platform === 'GSC') {
                res = await oauthAPI.connectGSC(sId);
            } else {
                res = await oauthAPI.connectGA4(sId);
            }

            // Redirect to Google OAuth
            if (res.data.authorization_url) {
                window.location.href = res.data.authorization_url;
            } else {
                showToast('Failed to get authorization URL', 'error');
            }
        } catch (error) {
            console.error(error);
            showToast(`Failed to initiate ${platform} connection`, 'error');
        }
    };

    if (loading) return <Container className="p-5 text-center"><Spinner animation="border" /></Container>;
    if (!strategy) return <Container className="p-5">Strategy not found</Container>;

    const getStatusVariant = (status: string) => {
        if (status?.includes('Live')) return 'success';
        if (status === 'Pending') return 'primary';
        return 'warning';
    };

    const getIntegrationStatus = (platform: string) => {
        const int = strategy.integrations.find(i => i.platform === platform && i.is_connected);
        if (int) return {
            status: 'Connected',
            variant: 'success',
            detail: int.property_name || 'Authorized'
        };
        return { status: 'Not Connected', variant: 'secondary', detail: '' };
    };

    const gscStatus = getIntegrationStatus('GSC');
    const ga4Status = getIntegrationStatus('GA4');

    return (
        <Container fluid className="py-4">
            <div className="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 className="mb-0" style={{ color: 'var(--text-main)' }}>{strategy.name}</h1>
                    <Badge bg={getStatusVariant(strategy.status)} className="mt-2">{strategy.status}</Badge>
                </div>
                <div className="text-muted">
                    ID: {strategy.id} | Owner: {strategy.owner_name}
                </div>
            </div>

            <Row className="g-4">
                {/* Left Column: Details & Integrations */}
                <Col md={8}>
                    <Card className="card-custom mb-4">
                        <Card.Header>
                            <h5 className="mb-0">Strategy Overview</h5>
                        </Card.Header>
                        <Card.Body>
                            <p>{strategy.description}</p>
                            <Row className="mt-4">
                                <Col sm={6}>
                                    <strong>Go Live Date:</strong> <span className="text-muted">{strategy.go_live_date || 'N/A'}</span>
                                </Col>
                                <Col sm={6}>
                                    <strong>Test End Date:</strong> <span className="text-muted">{strategy.test_end_date || 'N/A'}</span>
                                </Col>
                            </Row>
                            <Row className="mt-2">
                                <Col sm={6}>
                                    <strong>Benchmark Start:</strong> <span className="text-muted">{strategy.benchmark_start_date || 'N/A'}</span>
                                </Col>
                                <Col sm={6}>
                                    <strong>Benchmark End:</strong> <span className="text-muted">{strategy.benchmark_end_date || 'N/A'}</span>
                                </Col>
                            </Row>
                        </Card.Body>
                    </Card>

                    <Card className="card-custom">
                        <Card.Header>
                            <h5 className="mb-0">Integrations (OAuth)</h5>
                        </Card.Header>
                        <Card.Body>
                            <Row>
                                <Col md={6} className="mb-3 mb-md-0">
                                    <Card className="h-100 bg-transparent border-secondary">
                                        <Card.Body>
                                            <div className="d-flex justify-content-between align-items-start">
                                                <div>
                                                    <h6>Google Search Console</h6>
                                                    <Badge bg={gscStatus.variant}>{gscStatus.status}</Badge>
                                                    {gscStatus.detail && <div className="small text-muted mt-1">{gscStatus.detail}</div>}
                                                </div>
                                                <Button
                                                    variant={gscStatus.status === 'Connected' ? 'outline-success' : 'outline-primary'}
                                                    size="sm"
                                                    onClick={() => handleConnect('GSC')}
                                                >
                                                    {gscStatus.status === 'Connected' ? 'Re-authorize' : 'Connect with Google'}
                                                </Button>
                                            </div>
                                        </Card.Body>
                                    </Card>
                                </Col>
                                <Col md={6}>
                                    <Card className="h-100 bg-transparent border-secondary">
                                        <Card.Body>
                                            <div className="d-flex justify-content-between align-items-start">
                                                <div>
                                                    <h6>Google Analytics 4</h6>
                                                    <Badge bg={ga4Status.variant}>{ga4Status.status}</Badge>
                                                    {ga4Status.detail && <div className="small text-muted mt-1">{ga4Status.detail}</div>}
                                                </div>
                                                <Button
                                                    variant={ga4Status.status === 'Connected' ? 'outline-success' : 'outline-primary'}
                                                    size="sm"
                                                    onClick={() => handleConnect('GA4')}
                                                >
                                                    {ga4Status.status === 'Connected' ? 'Re-authorize' : 'Connect with Google'}
                                                </Button>
                                            </div>
                                        </Card.Body>
                                    </Card>
                                </Col>
                            </Row>
                        </Card.Body>
                    </Card>
                </Col>

                {/* Right Column: Notes */}
                <Col md={4}>
                    <Card className="card-custom h-100">
                        <Card.Header className="d-flex justify-content-between align-items-center">
                            <h5 className="mb-0">Notes</h5>
                            <Badge bg="secondary" pill>{notes.length}</Badge>
                        </Card.Header>
                        <Card.Body className="d-flex flex-column">
                            <Form onSubmit={handleAddNote} className="mb-3">
                                <Form.Group className="mb-2">
                                    <Form.Control
                                        as="textarea"
                                        rows={2}
                                        placeholder="Add a note..."
                                        value={newNote}
                                        onChange={(e) => setNewNote(e.target.value)}
                                        className="form-control-custom"
                                    />
                                </Form.Group>
                                <div className="text-end">
                                    <Button type="submit" variant="primary" size="sm" disabled={submittingNote}>
                                        {submittingNote ? 'Adding...' : 'Add Note'}
                                    </Button>
                                </div>
                            </Form>

                            <div className="flex-grow-1 overflow-auto" style={{ maxHeight: '500px' }}>
                                <ListGroup variant="flush">
                                    {notes.length === 0 && <div className="text-center text-muted mt-4">No notes yet.</div>}
                                    {notes.map(note => (
                                        <ListGroup.Item key={note.id} className="bg-transparent text-light border-secondary px-0">
                                            <div className="d-flex justify-content-between small text-muted mb-1">
                                                <span>{new Date(note.created_at).toLocaleString()}</span>
                                            </div>
                                            <div style={{ whiteSpace: 'pre-wrap' }}>{note.content}</div>
                                        </ListGroup.Item>
                                    ))}
                                </ListGroup>
                            </div>
                        </Card.Body>
                    </Card>
                </Col>
            </Row>
        </Container>
    );
};

export default StrategyDetail;
