import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Container from 'react-bootstrap/Container';
import Card from 'react-bootstrap/Card';
import Form from 'react-bootstrap/Form';
import Button from 'react-bootstrap/Button';
import ProgressBar from 'react-bootstrap/ProgressBar';
import Row from 'react-bootstrap/Row';
import Col from 'react-bootstrap/Col';
import Alert from 'react-bootstrap/Alert';
import { strategyAPI, oauthAPI } from '../services/api';
import { useSearchParams } from 'react-router-dom';

interface FormData {
    // Step 1
    name: string;
    description: string;
    owner_name: string;
    go_live_date: string;
    test_end_date: string;
    monitored_urls: string;

    // Step 2
    benchmark_start_date: string;
    benchmark_end_date: string;
    update_frequency_hours: number;
    skip_connections: boolean;

    // Step 3
    uploaded_file: File | null;
    data_type: string;

    // Step 4
    target_kpis: {
        clicks: string;
        impressions: string;
        sessions: string;
        conversions: string;
    };
}

const StrategyWizard: React.FC = () => {
    const navigate = useNavigate();
    const [currentStep, setCurrentStep] = useState(1);
    const [loading, setLoading] = useState(false);
    const [formData, setFormData] = useState<FormData>({
        name: '',
        description: '',
        owner_name: 'Daine Dvorak',
        go_live_date: '',
        test_end_date: '',
        monitored_urls: '',
        benchmark_start_date: '',
        benchmark_end_date: '',
        update_frequency_hours: 24,
        skip_connections: false,
        uploaded_file: null,
        data_type: 'benchmark',
        target_kpis: {
            clicks: '',
            impressions: '',
            sessions: '',
            conversions: ''
        }
    });

    // OAuth Integration State
    const [searchParams] = useSearchParams();
    const [strategyId, setStrategyId] = useState<number | null>(null);
    const [integrations, setIntegrations] = useState<any[]>([]);
    const [gscProperties, setGscProperties] = useState<any[]>([]);
    const [ga4Properties, setGa4Properties] = useState<any[]>([]);

    // Load strategy if ID is in URL (e.g. from OAuth callback)
    React.useEffect(() => {
        const sId = searchParams.get('strategy_id');
        const step = searchParams.get('step');

        if (sId) {
            const id = parseInt(sId);
            setStrategyId(id);
            if (step) setCurrentStep(parseInt(step));

            // Fetch strategy
            strategyAPI.get(id).then(res => {
                const s = res.data;
                setIntegrations(s.integrations || []);

                // Populate basic form data
                setFormData(prev => ({
                    ...prev,
                    name: s.name,
                    description: s.description,
                    owner_name: s.owner_name,
                    go_live_date: s.go_live_date || '',
                    test_end_date: s.test_end_date || '',
                    benchmark_start_date: s.benchmark_start_date || '',
                    benchmark_end_date: s.benchmark_end_date || '',
                    update_frequency_hours: s.update_frequency_hours,
                    monitored_urls: Array.isArray(s.monitored_urls) ? s.monitored_urls.join('\n') : '',
                    target_kpis: {
                        clicks: s.target_kpis?.clicks || '',
                        impressions: s.target_kpis?.impressions || '',
                        sessions: s.target_kpis?.sessions || '',
                        conversions: s.target_kpis?.conversions || ''
                    }
                }));

                // If on step 2, fetch properties for connected integrations
                if (currentStep === 2 || parseInt(step || '1') === 2) {
                    loadProperties(id, s.integrations);
                }
            }).catch(console.error);
        }
    }, [searchParams]);

    const loadProperties = async (id: number, currentIntegrations: any[]) => {
        const gscInt = currentIntegrations.find((i: any) => i.platform === 'GCS' || i.platform === 'GSC');
        const ga4Int = currentIntegrations.find((i: any) => i.platform === 'GA4');

        if (gscInt && gscInt.is_connected) {
            try {
                const res = await oauthAPI.fetchProperties('GSC', id);
                setGscProperties(res.data);
            } catch (e) {
                console.error('Failed to load GSC properties', e);
            }
        }

        if (ga4Int && ga4Int.is_connected) {
            try {
                const res = await oauthAPI.fetchProperties('GA4', id);
                setGa4Properties(res.data);
            } catch (e) {
                console.error('Failed to load GA4 properties', e);
            }
        }
    };

    const totalSteps = 4;
    const progress = (currentStep / totalSteps) * 100;

    const handleNext = async () => {
        if (validateStep(currentStep)) {

            // Save Logic for Step 1
            if (currentStep === 1) {
                setLoading(true);
                try {
                    const payload = getPayload();
                    let response;

                    if (strategyId) {
                        response = await strategyAPI.update(strategyId, payload);
                    } else {
                        response = await strategyAPI.create(payload);
                        setStrategyId(response.data.id);
                    }

                    // Update loaded integrations in case they changed (unlikely here but good practice)
                    setIntegrations(response.data.integrations || []);

                } catch (error) {
                    console.error('Failed to save strategy:', error);
                    alert('Failed to save progress. Please try again.');
                    setLoading(false);
                    return; // Stop navigation
                }
                setLoading(false);
            }

            // Initial Load for properties when entering Step 2
            if (currentStep === 1 && strategyId) {
                // Assuming we just saved and setStrategyId might not be updated inside this closure yet 
                // actually it updates on render. Use response.data.id if needed, but we can rely on effect or next render?
                // No, we need to ensure Step 2 loads correct data. The Effect handles data loading on mount, 
                // but here we are transitioning.
            }

            setCurrentStep(currentStep + 1);
        }
    };

    // Helper to extract payload
    const getPayload = () => {
        const urlsArray = formData.monitored_urls
            .split('\n')
            .map(url => url.trim())
            .filter(url => url.length > 0);

        return {
            name: formData.name,
            description: formData.description,
            owner_name: formData.owner_name,
            go_live_date: formData.go_live_date,
            test_end_date: formData.test_end_date,
            benchmark_start_date: formData.benchmark_start_date,
            benchmark_end_date: formData.benchmark_end_date,
            update_frequency_hours: formData.update_frequency_hours,
            monitored_urls: urlsArray,
            target_kpis: formData.target_kpis
        };
    };

    const handleConnect = async (platform: 'GSC' | 'GA4') => {
        if (!strategyId) return;
        try {
            const res = platform === 'GSC'
                ? await oauthAPI.connectGSC(strategyId)
                : await oauthAPI.connectGA4(strategyId);
            if (res.data.authorization_url) {
                window.location.href = res.data.authorization_url;
            }
        } catch (error) {
            console.error('Connection failed:', error);
            alert('Failed to initiate connection');
        }
    };

    const handlePropertySelect = async (platform: 'GSC' | 'GA4', propertyId: string) => {
        if (!strategyId) return;
        try {
            const property = (platform === 'GSC' ? gscProperties : ga4Properties)
                .find(p => p.id === propertyId);

            await strategyAPI.updateIntegration(strategyId, {
                platform,
                property_id: propertyId,
                property_name: property?.name
            });

            // Refresh integrations
            const res = await strategyAPI.get(strategyId);
            setIntegrations(res.data.integrations || []);

        } catch (error) {
            console.error('Failed to update property:', error);
        }
    };

    const handleBack = () => {
        setCurrentStep(currentStep - 1);
    };

    const validateStep = (step: number): boolean => {
        switch (step) {
            case 1:
                if (!formData.name || !formData.description) {
                    alert('Please fill in all required fields');
                    return false;
                }
                return true;
            case 2:
                // Optional validation for step 2
                return true;
            case 3:
                // Optional validation for step 3
                return true;
            case 4:
                return true;
            default:
                return true;
        }
    };

    const handleSubmit = async () => {
        setLoading(true);
        try {
            const payload = getPayload();

            if (strategyId) {
                await strategyAPI.update(strategyId, payload);
                navigate(`/strategy/${strategyId}`);
            } else {
                const response = await strategyAPI.create(payload);
                navigate(`/strategy/${response.data.id}`);
            }
        } catch (error) {
            console.error('Failed to create strategy:', error);
            alert('Failed to create strategy');
            setLoading(false);
        }
    };

    const renderStep = () => {
        switch (currentStep) {
            case 1:
                return (
                    <>
                        <h4 className="mb-4">Step 1: Basic Strategy Information</h4>

                        <Form.Group className="mb-3">
                            <Form.Label>Strategy Name *</Form.Label>
                            <Form.Control
                                type="text"
                                className="form-control-custom"
                                placeholder="e.g., Q1 2026 Keyword Dominance"
                                value={formData.name}
                                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                required
                            />
                        </Form.Group>

                        <Form.Group className="mb-3">
                            <Form.Label>Description *</Form.Label>
                            <Form.Control
                                as="textarea"
                                rows={4}
                                className="form-control-custom"
                                placeholder="Describe the goals and scope of this strategy..."
                                value={formData.description}
                                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                                required
                            />
                        </Form.Group>

                        <Form.Group className="mb-3">
                            <Form.Label>Owner</Form.Label>
                            <Form.Control
                                type="text"
                                className="form-control-custom"
                                value={formData.owner_name}
                                onChange={(e) => setFormData({ ...formData, owner_name: e.target.value })}
                            />
                        </Form.Group>

                        <Row>
                            <Col md={6}>
                                <Form.Group className="mb-3">
                                    <Form.Label>Go Live Date</Form.Label>
                                    <Form.Control
                                        type="date"
                                        className="form-control-custom"
                                        value={formData.go_live_date}
                                        onChange={(e) => setFormData({ ...formData, go_live_date: e.target.value })}
                                    />
                                </Form.Group>
                            </Col>
                            <Col md={6}>
                                <Form.Group className="mb-3">
                                    <Form.Label>Test End Date</Form.Label>
                                    <Form.Control
                                        type="date"
                                        className="form-control-custom"
                                        value={formData.test_end_date}
                                        onChange={(e) => setFormData({ ...formData, test_end_date: e.target.value })}
                                    />
                                </Form.Group>
                            </Col>
                        </Row>

                        <Form.Group className="mb-3">
                            <Form.Label>Monitored URLs (one per line)</Form.Label>
                            <Form.Control
                                as="textarea"
                                rows={5}
                                className="form-control-custom"
                                placeholder="https://example.com/page1&#10;https://example.com/page2"
                                value={formData.monitored_urls}
                                onChange={(e) => setFormData({ ...formData, monitored_urls: e.target.value })}
                                style={{ fontFamily: 'monospace' }}
                            />
                        </Form.Group>
                    </>
                );

            case 2:
                return (
                    <>
                        <h4 className="mb-4">Step 2: Benchmark Data & Data Sources</h4>

                        <Alert variant="info">
                            <strong>📊 Benchmark Period:</strong> Set the date range for your baseline comparison data. This will be used to measure the effectiveness of your strategy.
                        </Alert>

                        <Row className="mb-4">
                            <Col md={6}>
                                <Form.Group className="mb-3">
                                    <Form.Label>Benchmark Start Date</Form.Label>
                                    <Form.Control
                                        type="date"
                                        className="form-control-custom"
                                        value={formData.benchmark_start_date}
                                        onChange={(e) => setFormData({ ...formData, benchmark_start_date: e.target.value })}
                                    />
                                </Form.Group>
                            </Col>
                            <Col md={6}>
                                <Form.Group className="mb-3">
                                    <Form.Label>Benchmark End Date</Form.Label>
                                    <Form.Control
                                        type="date"
                                        className="form-control-custom"
                                        value={formData.benchmark_end_date}
                                        onChange={(e) => setFormData({ ...formData, benchmark_end_date: e.target.value })}
                                    />
                                </Form.Group>
                            </Col>
                        </Row>

                        <Form.Group className="mb-4">
                            <Form.Label>Data Update Frequency</Form.Label>
                            <Form.Select
                                className="form-control-custom"
                                value={formData.update_frequency_hours}
                                onChange={(e) => setFormData({ ...formData, update_frequency_hours: parseInt(e.target.value) })}
                            >
                                <option value="12">Every 12 hours</option>
                                <option value="24">Every 24 hours (default)</option>
                                <option value="48">Every 48 hours</option>
                                <option value="168">Weekly</option>
                            </Form.Select>
                            <Form.Text className="text-muted">
                                How often should we fetch new data from GSC and GA4?
                            </Form.Text>
                        </Form.Group>

                        <Card className="card-custom mb-3">
                            <Card.Body>
                                <h6 className="mb-3">Connect Data Sources</h6>
                                <p className="text-muted small">Connect GSC and GA4 to automatically import data for this test.</p>

                                <div className="d-grid gap-2 mb-3">
                                    {/* Buttons replaced by detailed sections above */}
                                </div>

                                <Form.Check
                                    type="checkbox"
                                    id="skip-connections"
                                    label="Skip for now - I'll upload data manually later"
                                    checked={formData.skip_connections}
                                    onChange={(e) => setFormData({ ...formData, skip_connections: e.target.checked })}
                                />
                            </Card.Body>
                        </Card>

                        <Alert variant="info">
                            <small>Connect your Google accounts to import benchmark data directly.</small>
                        </Alert>

                        {/* GSC Connection */}
                        <div className="mb-3 p-3 border rounded">
                            <h6>Google Search Console</h6>
                            {integrations.find(i => i.platform === 'GSC' && i.is_connected) ? (
                                <div className="text-success">
                                    ✓ Connected
                                    <Form.Select
                                        className="mt-2"
                                        value={integrations.find(i => i.platform === 'GSC')?.property_id || ''}
                                        onChange={(e) => handlePropertySelect('GSC', e.target.value)}
                                        onClick={() => {
                                            if (gscProperties.length === 0 && strategyId) loadProperties(strategyId, integrations);
                                        }}
                                    >
                                        <option value="">Select a Site...</option>
                                        {gscProperties.map(p => (
                                            <option key={p.id} value={p.id}>{p.name}</option>
                                        ))}
                                    </Form.Select>
                                </div>
                            ) : (
                                <Button variant="outline-primary" onClick={() => handleConnect('GSC')} disabled={!strategyId}>
                                    🔗 Connect Google Search Console
                                </Button>
                            )}
                        </div>

                        {/* GA4 Connection */}
                        <div className="mb-3 p-3 border rounded">
                            <h6>Google Analytics 4</h6>
                            {integrations.find(i => i.platform === 'GA4' && i.is_connected) ? (
                                <div className="text-success">
                                    ✓ Connected
                                    <Form.Select
                                        className="mt-2"
                                        value={integrations.find(i => i.platform === 'GA4')?.property_id || ''}
                                        onChange={(e) => handlePropertySelect('GA4', e.target.value)}
                                        onClick={() => {
                                            if (ga4Properties.length === 0 && strategyId) loadProperties(strategyId, integrations);
                                        }}
                                    >
                                        <option value="">Select a Property...</option>
                                        {ga4Properties.map(p => (
                                            <option key={p.id} value={p.id}>{p.name}</option>
                                        ))}
                                    </Form.Select>
                                </div>
                            ) : (
                                <Button variant="outline-primary" onClick={() => handleConnect('GA4')} disabled={!strategyId}>
                                    🔗 Connect Google Analytics 4
                                </Button>
                            )}
                        </div>
                    </>
                );

            case 3:
                return (
                    <>
                        <h4 className="mb-4">Step 3: File Upload (Optional)</h4>

                        <Alert variant="info">
                            <strong>📁 Upload Data:</strong> You can upload CSV or Excel files with your benchmark or performance data.
                        </Alert>

                        <Card className="card-custom mb-4">
                            <Card.Body>
                                <h6 className="mb-3">Download Template</h6>
                                <p className="text-muted small">Download a template file to see the required format.</p>
                                <Button variant="outline-secondary" size="sm" disabled>
                                    ⬇️ Download CSV Template
                                </Button>
                            </Card.Body>
                        </Card>

                        <Form.Group className="mb-3">
                            <Form.Label>Data Type</Form.Label>
                            <div>
                                <Form.Check
                                    type="radio"
                                    id="data-type-benchmark"
                                    name="data_type"
                                    label="Benchmark Data (baseline for comparison)"
                                    value="benchmark"
                                    checked={formData.data_type === 'benchmark'}
                                    onChange={(e) => setFormData({ ...formData, data_type: e.target.value })}
                                    className="mb-2"
                                />
                                <Form.Check
                                    type="radio"
                                    id="data-type-performance"
                                    name="data_type"
                                    label="Post-Performance Data (live test results)"
                                    value="performance"
                                    checked={formData.data_type === 'performance'}
                                    onChange={(e) => setFormData({ ...formData, data_type: e.target.value })}
                                />
                            </div>
                            <Form.Text className="text-muted">
                                The system auto-selects based on strategy status. Override if needed.
                            </Form.Text>
                        </Form.Group>

                        <Form.Group className="mb-3">
                            <Form.Label>Upload File</Form.Label>
                            <Form.Control
                                type="file"
                                className="form-control-custom"
                                accept=".csv,.xlsx,.xls"
                                onChange={(e) => {
                                    const files = (e.target as HTMLInputElement).files;
                                    setFormData({ ...formData, uploaded_file: files ? files[0] : null });
                                }}
                            />
                            <Form.Text className="text-muted">
                                Accepted formats: CSV, Excel (.xlsx, .xls)
                            </Form.Text>
                        </Form.Group>
                    </>
                );

            case 4:
                return (
                    <>
                        <h4 className="mb-4">Step 4: Target KPIs & Review</h4>

                        <Alert variant="success">
                            <strong>✅ Almost Done!</strong> Set your target KPIs and review your strategy before submitting.
                        </Alert>

                        <Card className="card-custom mb-4">
                            <Card.Header>
                                <h6 className="mb-0">Target KPIs (Monthly)</h6>
                            </Card.Header>
                            <Card.Body>
                                <Row>
                                    <Col md={6}>
                                        <Form.Group className="mb-3">
                                            <Form.Label>Target Clicks</Form.Label>
                                            <Form.Control
                                                type="number"
                                                className="form-control-custom"
                                                placeholder="e.g., 1000"
                                                value={formData.target_kpis.clicks}
                                                onChange={(e) => setFormData({
                                                    ...formData,
                                                    target_kpis: { ...formData.target_kpis, clicks: e.target.value }
                                                })}
                                            />
                                        </Form.Group>
                                    </Col>
                                    <Col md={6}>
                                        <Form.Group className="mb-3">
                                            <Form.Label>Target Impressions</Form.Label>
                                            <Form.Control
                                                type="number"
                                                className="form-control-custom"
                                                placeholder="e.g., 50000"
                                                value={formData.target_kpis.impressions}
                                                onChange={(e) => setFormData({
                                                    ...formData,
                                                    target_kpis: { ...formData.target_kpis, impressions: e.target.value }
                                                })}
                                            />
                                        </Form.Group>
                                    </Col>
                                    <Col md={6}>
                                        <Form.Group className="mb-3">
                                            <Form.Label>Target Sessions</Form.Label>
                                            <Form.Control
                                                type="number"
                                                className="form-control-custom"
                                                placeholder="e.g., 5000"
                                                value={formData.target_kpis.sessions}
                                                onChange={(e) => setFormData({
                                                    ...formData,
                                                    target_kpis: { ...formData.target_kpis, sessions: e.target.value }
                                                })}
                                            />
                                        </Form.Group>
                                    </Col>
                                    <Col md={6}>
                                        <Form.Group className="mb-3">
                                            <Form.Label>Target Conversions</Form.Label>
                                            <Form.Control
                                                type="number"
                                                className="form-control-custom"
                                                placeholder="e.g., 50"
                                                value={formData.target_kpis.conversions}
                                                onChange={(e) => setFormData({
                                                    ...formData,
                                                    target_kpis: { ...formData.target_kpis, conversions: e.target.value }
                                                })}
                                            />
                                        </Form.Group>
                                    </Col>
                                </Row>
                            </Card.Body>
                        </Card>

                        <Card className="card-custom">
                            <Card.Header>
                                <h6 className="mb-0">Review Your Strategy</h6>
                            </Card.Header>
                            <Card.Body>
                                <dl className="row mb-0">
                                    <dt className="col-sm-4">Name:</dt>
                                    <dd className="col-sm-8">{formData.name || '—'}</dd>

                                    <dt className="col-sm-4">Owner:</dt>
                                    <dd className="col-sm-8">{formData.owner_name || '—'}</dd>

                                    <dt className="col-sm-4">Go Live:</dt>
                                    <dd className="col-sm-8">{formData.go_live_date || '—'}</dd>

                                    <dt className="col-sm-4">Benchmark Period:</dt>
                                    <dd className="col-sm-8">
                                        {formData.benchmark_start_date && formData.benchmark_end_date
                                            ? `${formData.benchmark_start_date} to ${formData.benchmark_end_date}`
                                            : '—'}
                                    </dd>

                                    <dt className="col-sm-4">Update Frequency:</dt>
                                    <dd className="col-sm-8">Every {formData.update_frequency_hours} hours</dd>

                                    <dt className="col-sm-4">Monitored URLs:</dt>
                                    <dd className="col-sm-8">
                                        {formData.monitored_urls.split('\n').filter(u => u.trim()).length || 0} URLs
                                    </dd>
                                </dl>
                            </Card.Body>
                        </Card>
                    </>
                );

            default:
                return null;
        }
    };

    return (
        <Container fluid style={{ maxWidth: '900px' }}>
            <div className="mb-4">
                <Button variant="link" onClick={() => navigate('/')} style={{ color: 'var(--text-muted)', textDecoration: 'none' }}>
                    ← Back to Dashboard
                </Button>
            </div>

            <Card className="card-custom">
                <Card.Body className="p-4">
                    {/* Progress Bar */}
                    <div className="mb-4">
                        <div className="d-flex justify-content-between mb-2">
                            <span style={{ color: 'var(--text-muted)', fontSize: '0.875rem' }}>
                                Step {currentStep} of {totalSteps}
                            </span>
                            <span style={{ color: 'var(--text-muted)', fontSize: '0.875rem' }}>
                                {Math.round(progress)}% Complete
                            </span>
                        </div>
                        <ProgressBar now={progress} variant="primary" />
                    </div>

                    {/* Step Content */}
                    {renderStep()}

                    {/* Navigation Buttons */}
                    <div className="d-flex justify-content-between mt-4 pt-4" style={{ borderTop: '1px solid var(--border)' }}>
                        <Button
                            variant="outline-secondary"
                            onClick={handleBack}
                            disabled={currentStep === 1}
                        >
                            ← Back
                        </Button>

                        {currentStep < totalSteps ? (
                            <Button variant="primary" className="btn-primary-custom" onClick={handleNext}>
                                Next Step →
                            </Button>
                        ) : (
                            <Button
                                variant="success"
                                onClick={handleSubmit}
                                disabled={loading}
                            >
                                {loading ? 'Creating...' : '✓ Create Strategy'}
                            </Button>
                        )}
                    </div>
                </Card.Body>
            </Card>
        </Container>
    );
};

export default StrategyWizard;
