import axios from 'axios';

// Create axios instance with base configuration
const api = axios.create({
    baseURL: '/v3/api',
    headers: {
        'Content-Type': 'application/json'
    }
});

// Strategy API
export const strategyAPI = {
    list: (status?: string) =>
        api.get('/strategies', { params: status ? { status } : {} }),

    create: (data: any) =>
        api.post('/strategies', data),

    get: (id: number) =>
        api.get(`/strategies/${id}`),

    update: (id: number, data: any) =>
        api.put(`/strategies/${id}`, data),

    delete: (id: number) =>
        api.delete(`/strategies/${id}`),

    getStatus: (id: number) =>
        api.get(`/strategies/${id}/status`),

    getNotes: (id: number) =>
        api.get(`/strategies/${id}/notes`),

    addNote: (id: number, content: string) =>
        api.post(`/strategies/${id}/notes`, { content }),

    updateIntegration: (id: number, data: any) =>
        api.post(`/strategies/${id}/integrations`, data)
};

// OAuth API
export const oauthAPI = {
    connectGSC: (strategyId: number) =>
        api.post(`/oauth/connect/gsc`, { strategy_id: strategyId }),

    connectGA4: (strategyId: number) =>
        api.post(`/oauth/connect/ga4`, { strategy_id: strategyId }),

    fetchProperties: (platform: string, strategyId: number) =>
        api.get(`/oauth/${platform}/properties`, { params: { strategy_id: strategyId } }),
};

// Data API
export const dataAPI = {
    importData: (strategyId: number, source: string) =>
        api.post(`/data/import`, { strategy_id: strategyId, source }),

    uploadFile: (strategyId: number, file: File, dataType: string) => {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('data_type', dataType);
        return api.post(`/data/upload/${strategyId}`, formData, {
            headers: { 'Content-Type': 'multipart/form-data' }
        });
    }
};

// Settings API
export const settingsAPI = {
    get: () => api.get('/settings'),
    update: (data: any) => api.put('/settings', data)
};

// Template API
export const templateAPI = {
    download: (type: string) =>
        api.get(`/templates/download/${type}`, { responseType: 'blob' })
};

export const docsAPI = {
    list: () => api.get<string[]>('/docs'),
    get: (filename: string) => api.get<{ filename: string, content: string }>(`/docs/${filename}`),
};

export default api;
