from flask import Flask
from flask_sqlalchemy import SQLAlchemy
from flask_migrate import Migrate
from flask_cors import CORS
import os

db = SQLAlchemy()
migrate = Migrate()

def create_app():
    app = Flask(__name__)
    
    # Configuration
    app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', 'dev-secret-key-change-in-production')
    basedir = os.path.abspath(os.path.dirname(__file__))
    app.config['SQLALCHEMY_DATABASE_URI'] = os.environ.get('DATABASE_URL', f'sqlite:///{os.path.join(basedir, "../instance/v3.db")}')
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    app.config['UPLOAD_FOLDER'] = os.path.join(basedir, '../uploads')
    app.config['MAX_CONTENT_LENGTH'] = 16 * 1024 * 1024  # 16MB max file size
    
    # Ensure upload folder exists
    os.makedirs(app.config['UPLOAD_FOLDER'], exist_ok=True)
    
    # Initialize extensions
    db.init_app(app)
    migrate.init_app(app, db)
    CORS(app)
    
    # Register blueprints
    from app.routes import strategy_bp, oauth_bp, data_bp, settings_bp, template_bp, docs_bp
    app.register_blueprint(strategy_bp, url_prefix='/api/strategies')
    app.register_blueprint(oauth_bp, url_prefix='/api/oauth')
    app.register_blueprint(data_bp, url_prefix='/api/data')
    app.register_blueprint(settings_bp, url_prefix='/api/settings')
    app.register_blueprint(template_bp, url_prefix='/api/templates')
    app.register_blueprint(docs_bp, url_prefix='/api/docs')
    
    return app
