from app import db
from datetime import datetime, date
from typing import Optional
import json


class Strategy(db.Model):
    """Core strategy/test entity with state tracking"""
    __tablename__ = 'strategy'
    
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(200), nullable=False)
    description = db.Column(db.Text)
    owner_id = db.Column(db.Integer, nullable=True)  # Future: link to User model
    owner_name = db.Column(db.String(100))
    
    # Dates
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    go_live_date = db.Column(db.Date, nullable=True)
    test_end_date = db.Column(db.Date, nullable=True)
    
    # Benchmark period
    benchmark_start_date = db.Column(db.Date, nullable=True)
    benchmark_end_date = db.Column(db.Date, nullable=True)
    
    # Configuration
    update_frequency_hours = db.Column(db.Integer, default=24)
    monitored_urls = db.Column(db.JSON)  # Array of URLs
    target_kpis = db.Column(db.JSON)  # Dict of KPI targets
    
    # Status (computed property, but we can cache it)
    status_cache = db.Column(db.String(50))  # Planning, Pending, Live
    
    # Relationships
    integrations = db.relationship('Integration', backref='strategy', lazy='dynamic', cascade='all, delete-orphan')
    benchmark_data = db.relationship('BenchmarkData', backref='strategy', lazy='dynamic', cascade='all, delete-orphan')
    performance_data = db.relationship('PerformanceData', backref='strategy', lazy='dynamic', cascade='all, delete-orphan')
    uploads = db.relationship('FileUpload', backref='strategy', lazy='dynamic', cascade='all, delete-orphan')
    notes = db.relationship('Note', backref='strategy', lazy='dynamic', cascade='all, delete-orphan')
    
    def calculate_status(self):
        """Calculate current strategy status based on business rules"""
        # Check required fields
        required_fields = [self.name, self.description, self.go_live_date]
        has_all_required = all(required_fields)
        has_benchmark_data = self.benchmark_data.count() > 0
        
        if not has_all_required:
            return 'Planning'
        
        if has_all_required and has_benchmark_data:
            if self.go_live_date and self.go_live_date > date.today():
                return 'Pending'
            elif self.go_live_date and self.go_live_date <= date.today():
                return 'Live - Collecting Live Test Data'
        
        return 'Planning'
    
    def to_dict(self):
        """Serialize strategy to dictionary"""
        return {
            'id': self.id,
            'name': self.name,
            'description': self.description,
            'owner_name': self.owner_name,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'go_live_date': self.go_live_date.isoformat() if self.go_live_date else None,
            'test_end_date': self.test_end_date.isoformat() if self.test_end_date else None,
            'benchmark_start_date': self.benchmark_start_date.isoformat() if self.benchmark_start_date else None,
            'benchmark_end_date': self.benchmark_end_date.isoformat() if self.benchmark_end_date else None,
            'update_frequency_hours': self.update_frequency_hours,
            'monitored_urls': self.monitored_urls,
            'target_kpis': self.target_kpis,
            'status': self.calculate_status(),
            'integrations': [i.to_dict() for i in self.integrations],
            'benchmark_data_count': self.benchmark_data.count(),
            'performance_data_count': self.performance_data.count()
        }


class Integration(db.Model):
    """Per-test OAuth connections to GSC/GA4"""
    __tablename__ = 'integration'
    
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    
    # Platform details
    platform = db.Column(db.String(20), nullable=False)  # 'GSC' or 'GA4'
    property_id = db.Column(db.String(500))  # Site URL for GSC, Property ID for GA4
    property_name = db.Column(db.String(500))  # Display name
    
    # OAuth tokens (encrypted in production!)
    access_token = db.Column(db.Text)
    refresh_token = db.Column(db.Text)
    token_expiry = db.Column(db.DateTime)
    
    # Configuration
    selected_metrics = db.Column(db.JSON)  # Array of metric names for GA4
    
    # Timestamps
    connected_at = db.Column(db.DateTime, default=datetime.utcnow)
    last_synced = db.Column(db.DateTime)
    
    def to_dict(self):
        return {
            'id': self.id,
            'platform': self.platform,
            'property_id': self.property_id,
            'property_name': self.property_name,
            'selected_metrics': self.selected_metrics,
            'connected_at': self.connected_at.isoformat() if self.connected_at else None,
            'last_synced': self.last_synced.isoformat() if self.last_synced else None,
            'is_connected': bool(self.access_token)
        }


class BenchmarkData(db.Model):
    """Baseline performance data for comparison"""
    __tablename__ = 'benchmark_data'
    
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    
    # Source tracking
    source = db.Column(db.String(20), nullable=False)  # 'GSC', 'GA4', 'Upload'
    date = db.Column(db.Date, nullable=False)
    url = db.Column(db.String(1000), nullable=False)
    
    # GSC Metrics
    clicks = db.Column(db.Integer, default=0)
    impressions = db.Column(db.Integer, default=0)
    ctr = db.Column(db.Float, default=0.0)
    position = db.Column(db.Float, default=0.0)
    
    # GA4 Metrics
    sessions = db.Column(db.Integer, default=0)
    pageviews = db.Column(db.Integer, default=0)
    users = db.Column(db.Integer, default=0)
    avg_session_duration = db.Column(db.Float, default=0.0)
    bounce_rate = db.Column(db.Float, default=0.0)
    conversions = db.Column(db.Integer, default=0)
    
    # Custom metrics (JSON for flexibility)
    custom_metrics = db.Column(db.JSON)
    
    # Metadata
    imported_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    __table_args__ = (
        db.Index('idx_benchmark_strategy_date', 'strategy_id', 'date'),
        db.Index('idx_benchmark_url', 'url'),
    )
    
    def to_dict(self):
        return {
            'id': self.id,
            'source': self.source,
            'date': self.date.isoformat() if self.date else None,
            'url': self.url,
            'clicks': self.clicks,
            'impressions': self.impressions,
            'ctr': self.ctr,
            'position': self.position,
            'sessions': self.sessions,
            'pageviews': self.pageviews,
            'users': self.users,
            'avg_session_duration': self.avg_session_duration,
            'bounce_rate': self.bounce_rate,
            'conversions': self.conversions,
            'custom_metrics': self.custom_metrics
        }


class PerformanceData(db.Model):
    """Live test performance data"""
    __tablename__ = 'performance_data'
    
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    
    # Source tracking
    source = db.Column(db.String(20), nullable=False)  # 'GSC', 'GA4', 'Upload'
    date = db.Column(db.Date, nullable=False)
    url = db.Column(db.String(1000), nullable=False)
    
    # Same metrics as BenchmarkData
    clicks = db.Column(db.Integer, default=0)
    impressions = db.Column(db.Integer, default=0)
    ctr = db.Column(db.Float, default=0.0)
    position = db.Column(db.Float, default=0.0)
    sessions = db.Column(db.Integer, default=0)
    pageviews = db.Column(db.Integer, default=0)
    users = db.Column(db.Integer, default=0)
    avg_session_duration = db.Column(db.Float, default=0.0)
    bounce_rate = db.Column(db.Float, default=0.0)
    conversions = db.Column(db.Integer, default=0)
    custom_metrics = db.Column(db.JSON)
    
    # Forecast flag
    is_forecast = db.Column(db.Boolean, default=False)
    
    # Metadata
    imported_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    __table_args__ = (
        db.Index('idx_performance_strategy_date', 'strategy_id', 'date'),
        db.Index('idx_performance_url', 'url'),
    )
    
    def to_dict(self):
        return {
            'id': self.id,
            'source': self.source,
            'date': self.date.isoformat() if self.date else None,
            'url': self.url,
            'clicks': self.clicks,
            'impressions': self.impressions,
            'ctr': self.ctr,
            'position': self.position,
            'sessions': self.sessions,
            'pageviews': self.pageviews,
            'users': self.users,
            'avg_session_duration': self.avg_session_duration,
            'bounce_rate': self.bounce_rate,
            'conversions': self.conversions,
            'custom_metrics': self.custom_metrics,
            'is_forecast': self.is_forecast
        }


class FileUpload(db.Model):
    """Track uploaded CSV/Excel files"""
    __tablename__ = 'file_upload'
    
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    
    filename = db.Column(db.String(500), nullable=False)
    data_type = db.Column(db.String(20), nullable=False)  # 'benchmark' or 'performance'
    file_path = db.Column(db.String(1000))
    
    # Processing metadata
    uploaded_at = db.Column(db.DateTime, default=datetime.utcnow)
    row_count = db.Column(db.Integer, default=0)
    processed_status = db.Column(db.String(20), default='pending')  # pending, success, error
    error_message = db.Column(db.Text)
    
    def to_dict(self):
        return {
            'id': self.id,
            'filename': self.filename,
            'data_type': self.data_type,
            'uploaded_at': self.uploaded_at.isoformat() if self.uploaded_at else None,
            'row_count': self.row_count,
            'processed_status': self.processed_status,
            'error_message': self.error_message
        }


class Settings(db.Model):
    """User preferences and configuration"""
    __tablename__ = 'settings'
    
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, default=1)  # Future: link to User model
    
    # Theme
    theme_variant = db.Column(db.Integer, default=1)  # 1-10
    
    # Validation settings
    required_fields_enabled = db.Column(db.Boolean, default=True)
    
    # Reporting
    advanced_reporting_enabled = db.Column(db.Boolean, default=False)
    default_chart_types = db.Column(db.JSON)  # Preferred chart types
    
    # Notification preferences
    email_notifications = db.Column(db.Boolean, default=True)
    update_notifications = db.Column(db.Boolean, default=True)
    
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    def to_dict(self):
        return {
            'theme_variant': self.theme_variant,
            'required_fields_enabled': self.required_fields_enabled,
            'advanced_reporting_enabled': self.advanced_reporting_enabled,
            'default_chart_types': self.default_chart_types,
            'email_notifications': self.email_notifications,
            'update_notifications': self.update_notifications
        }


class Note(db.Model):
    """Notes attached to a strategy"""
    __tablename__ = 'note'
    
    id = db.Column(db.Integer, primary_key=True)
    strategy_id = db.Column(db.Integer, db.ForeignKey('strategy.id'), nullable=False)
    
    content = db.Column(db.Text, nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    def to_dict(self):
        return {
            'id': self.id,
            'strategy_id': self.strategy_id,
            'content': self.content,
            'created_at': self.created_at.isoformat()
        }
