from flask import Blueprint, request, jsonify, redirect
from app import db
from app.models import Strategy
from datetime import datetime

strategy_bp = Blueprint('strategies', __name__)


@strategy_bp.route('', methods=['GET'])
def list_strategies():
    """List all strategies with optional status filter"""
    status_filter = request.args.get('status')
    
    strategies = Strategy.query.all()
    
    result = []
    for strategy in strategies:
        strategy_dict = strategy.to_dict()
        if status_filter and strategy_dict['status'] != status_filter:
            continue
        result.append(strategy_dict)
    
    return jsonify(result)


@strategy_bp.route('', methods=['POST'])
def create_strategy():
    """Create a new strategy from wizard"""
    data = request.json
    
    strategy = Strategy(
        name=data.get('name'),
        description=data.get('description'),
        owner_name=data.get('owner_name'),
        go_live_date=datetime.fromisoformat(data['go_live_date']).date() if data.get('go_live_date') else None,
        test_end_date=datetime.fromisoformat(data['test_end_date']).date() if data.get('test_end_date') else None,
        benchmark_start_date=datetime.fromisoformat(data['benchmark_start_date']).date() if data.get('benchmark_start_date') else None,
        benchmark_end_date=datetime.fromisoformat(data['benchmark_end_date']).date() if data.get('benchmark_end_date') else None,
        update_frequency_hours=data.get('update_frequency_hours', 24),
        monitored_urls=data.get('monitored_urls', []),
        target_kpis=data.get('target_kpis', {})
    )
    
    db.session.add(strategy)
    db.session.commit()
    
    return jsonify(strategy.to_dict()), 201


@strategy_bp.route('/<int:id>', methods=['GET'])
def get_strategy(id):
    """Get strategy details"""
    strategy = Strategy.query.get_or_404(id)
    return jsonify(strategy.to_dict())


@strategy_bp.route('/<int:id>', methods=['PUT'])
def update_strategy(id):
    """Update strategy"""
    strategy = Strategy.query.get_or_404(id)
    data = request.json
    
    # Update fields
    if 'name' in data:
        strategy.name = data['name']
    if 'description' in data:
        strategy.description = data['description']
    if 'owner_name' in data:
        strategy.owner_name = data['owner_name']
    if 'go_live_date' in data:
        strategy.go_live_date = datetime.fromisoformat(data['go_live_date']).date() if data['go_live_date'] else None
    if 'test_end_date' in data:
        strategy.test_end_date = datetime.fromisoformat(data['test_end_date']).date() if data['test_end_date'] else None
    if 'benchmark_start_date' in data:
        strategy.benchmark_start_date = datetime.fromisoformat(data['benchmark_start_date']).date() if data['benchmark_start_date'] else None
    if 'benchmark_end_date' in data:
        strategy.benchmark_end_date = datetime.fromisoformat(data['benchmark_end_date']).date() if data['benchmark_end_date'] else None
    if 'update_frequency_hours' in data:
        strategy.update_frequency_hours = data['update_frequency_hours']
    if 'monitored_urls' in data:
        strategy.monitored_urls = data['monitored_urls']
    if 'target_kpis' in data:
        strategy.target_kpis = data['target_kpis']
    
    db.session.commit()
    return jsonify(strategy.to_dict())


@strategy_bp.route('/<int:id>', methods=['DELETE'])
def delete_strategy(id):
    """Delete strategy"""
    strategy = Strategy.query.get_or_404(id)
    db.session.delete(strategy)
    db.session.commit()
    return '', 204


@strategy_bp.route('/<int:id>/status', methods=['GET'])
def get_strategy_status(id):
    """Get calculated strategy status"""
    strategy = Strategy.query.get_or_404(id)
    return jsonify({'status': strategy.calculate_status()})


@strategy_bp.route('/<int:id>/notes', methods=['GET', 'POST'])
def handle_notes(id):
    """Get or create notes for a strategy"""
    from app.models import Note
    strategy = Strategy.query.get_or_404(id)
    
    if request.method == 'GET':
        notes = strategy.notes.order_by(Note.created_at.desc()).all()
        return jsonify([n.to_dict() for n in notes])
    
    data = request.json
    note = Note(strategy_id=id, content=data['content'])
    db.session.add(note)
    db.session.commit()
    return jsonify(note.to_dict()), 201


@strategy_bp.route('/<int:id>/integrations', methods=['POST'])
def update_integration(id):
    """Add or update an integration for a strategy"""
    from app.models import Integration
    strategy = Strategy.query.get_or_404(id)
    data = request.json
    
    platform = data.get('platform')
    property_id = data.get('property_id')
    property_name = data.get('property_name')
    
    if not platform:
        return jsonify({'error': 'Platform is required'}), 400
        
    # Check if integration exists
    integration = Integration.query.filter_by(strategy_id=id, platform=platform).first()
    
    if not integration:
        integration = Integration(
            strategy_id=id,
            platform=platform
        )
        db.session.add(integration)
    
    if property_id:
        integration.property_id = property_id
    if property_name:
        integration.property_name = property_name
    
    # In a real oauth flow, tokens would be set here too. 
    # For now we allow manual property ID setting.
    
    db.session.commit()
    return jsonify(integration.to_dict())



# Additional blueprints (stubs for now)
oauth_bp = Blueprint('oauth', __name__)
data_bp = Blueprint('data', __name__)
settings_bp = Blueprint('settings', __name__)
template_bp = Blueprint('templates', __name__)


@oauth_bp.route('/connect/<platform>', methods=['POST'])
def connect_platform(platform):
    """Initiate OAuth connection"""
    from google_auth_oauthlib.flow import Flow
    import os
    
    platform = platform.upper()
    if platform not in ['GSC', 'GA4']:
        return jsonify({'error': 'Invalid platform'}), 400
        
    data = request.json
    strategy_id = data.get('strategy_id')
    
    if not strategy_id:
        return jsonify({'error': 'Strategy ID is required'}), 400
        
    # Define scopes based on platform
    SCOPES = []
    if platform == 'GSC':
        SCOPES = ['https://www.googleapis.com/auth/webmasters.readonly']
    elif platform == 'GA4':
        SCOPES = ['https://www.googleapis.com/auth/analytics.readonly']
        
    # Create flow
    flow = Flow.from_client_config(
        {
            "web": {
                "client_id": os.environ.get('GOOGLE_CLIENT_ID'),
                "client_secret": os.environ.get('GOOGLE_CLIENT_SECRET'),
                "auth_uri": "https://accounts.google.com/o/oauth2/auth",
                "token_uri": "https://oauth2.googleapis.com/token",
            }
        },
        scopes=SCOPES
    )
    
    flow.redirect_uri = os.environ.get('GOOGLE_REDIRECT_URI', 'http://localhost:5004/api/oauth/callback')
    
    authorization_url, state = flow.authorization_url(
        access_type='offline',
        include_granted_scopes='true',
        state=f"{platform}:{strategy_id}"
    )
    
    return jsonify({'authorization_url': authorization_url})


@oauth_bp.route('/callback', methods=['GET'])
def oauth_callback():
    """Handle OAuth callback"""
    from google_auth_oauthlib.flow import Flow
    from app.models import Integration
    import os
    
    state = request.args.get('state') # "platform:strategy_id"
    if not state or ':' not in state:
        return jsonify({'error': 'Invalid state parameter'}), 400
        
    platform, strategy_id = state.split(':')
    
    # Re-create flow (need scopes even for callback in some versions/configs if not strict)
    # Define scopes based on platform
    SCOPES = []
    if platform == 'GSC':
        SCOPES = ['https://www.googleapis.com/auth/webmasters.readonly']
    elif platform == 'GA4':
        SCOPES = ['https://www.googleapis.com/auth/analytics.readonly']

    flow = Flow.from_client_config(
        {
            "web": {
                "client_id": os.environ.get('GOOGLE_CLIENT_ID'),
                "client_secret": os.environ.get('GOOGLE_CLIENT_SECRET'),
                "auth_uri": "https://accounts.google.com/o/oauth2/auth",
                "token_uri": "https://oauth2.googleapis.com/token",
            }
        },
        scopes=SCOPES,
        state=state
    )
    flow.redirect_uri = os.environ.get('GOOGLE_REDIRECT_URI', 'http://localhost:5004/api/oauth/callback')
    
    # Fetch token
    authorization_response = request.url.replace('http:', 'https:') # Force HTTPS for OAuthlib (dev only)
    if 'localhost' in flow.redirect_uri:
         authorization_response = request.url # Allow http for localhost
         
    try:
        flow.fetch_token(authorization_response=authorization_response)
        credentials = flow.credentials
        
        # Save integration
        integration = Integration.query.filter_by(strategy_id=strategy_id, platform=platform).first()
        if not integration:
            integration = Integration(
                strategy_id=strategy_id,
                platform=platform
            )
            db.session.add(integration)
            
        integration.access_token = credentials.token
        integration.refresh_token = credentials.refresh_token
        integration.token_expiry = credentials.expiry
        integration.connected_at = datetime.utcnow()
        integration.property_name = f"Connected via OAuth ({datetime.now().strftime('%Y-%m-%d')})"
        
        db.session.commit()
        
        # Redirect back to frontend wizard
        frontend_url = os.environ.get('FRONTEND_URL', 'http://localhost:5173/v3')
        return redirect(f"{frontend_url}/wizard?strategy_id={strategy_id}&step=2&connected={platform}")
        
    except Exception as e:
        return jsonify({'error': str(e)}), 500


@oauth_bp.route('/<platform>/properties', methods=['GET'])
def list_properties(platform):
    """List available properties for a connected platform"""
    from app.models import Integration
    from google.oauth2.credentials import Credentials
    from googleapiclient.discovery import build
    import os
    
    platform = platform.upper()
    strategy_id = request.args.get('strategy_id')
    
    if not strategy_id:
        return jsonify({'error': 'Strategy ID is required'}), 400
        
    integration = Integration.query.filter_by(strategy_id=strategy_id, platform=platform).first()
    
    if not integration or not integration.access_token:
        return jsonify({'error': 'Platform not connected'}), 400
        
    # Build credentials
    creds = Credentials(
        token=integration.access_token,
        refresh_token=integration.refresh_token,
        token_uri="https://oauth2.googleapis.com/token",
        client_id=os.environ.get('GOOGLE_CLIENT_ID'),
        client_secret=os.environ.get('GOOGLE_CLIENT_SECRET'),
        scopes=['https://www.googleapis.com/auth/webmasters.readonly'] if platform == 'GSC' else ['https://www.googleapis.com/auth/analytics.readonly']
    )
    
    try:
        properties = []
        if platform == 'GSC':
            service = build('searchconsole', 'v1', credentials=creds)
            site_list = service.sites().list().execute()
            if 'siteEntry' in site_list:
                properties = [{'id': s['siteUrl'], 'name': s['siteUrl']} for s in site_list['siteEntry']]
                
        elif platform == 'GA4':
            service = build('analyticsadmin', 'v1beta', credentials=creds)
            # Use account summaries to get properties
            summaries = service.accountSummaries().list().execute()
            for account in summaries.get('accountSummaries', []):
                for prop in account.get('propertySummaries', []):
                    properties.append({
                        'id': prop['property'], # e.g. 'properties/123456'
                        'name': f"{account['displayName']} - {prop['displayName']}"
                    })
                    
        return jsonify(properties)
        
    except Exception as e:
        print(f"Error fetching properties: {e}")
        return jsonify({'error': str(e)}), 500


@data_bp.route('/import', methods=['POST'])
def import_data():
    """Import data from GSC/GA4 (stub)"""
    return jsonify({'message': 'Data import not yet implemented'}), 501


@settings_bp.route('', methods=['GET', 'PUT'])
def manage_settings():
    """Get or update settings"""
    from app.models import Settings
    
    if request.method == 'GET':
        settings = Settings.query.first()
        if not settings:
            settings = Settings()
            db.session.add(settings)
            db.session.commit()
        return jsonify(settings.to_dict())
    else:
        # Update settings
        data = request.json
        settings = Settings.query.first()
        if not settings:
            settings = Settings()
            db.session.add(settings)
        
        # Update fields from request data
        if 'theme_variant' in data:
            settings.theme_variant = data['theme_variant']
        if 'required_fields_enabled' in data:
            settings.required_fields_enabled = data['required_fields_enabled']
        if 'advanced_reporting_enabled' in data:
            settings.advanced_reporting_enabled = data['advanced_reporting_enabled']
        if 'email_notifications' in data:
            settings.email_notifications = data['email_notifications']
        if 'update_notifications' in data:
            settings.update_notifications = data['update_notifications']
        if 'default_chart_types' in data:
            settings.default_chart_types = data['default_chart_types']
        
        db.session.commit()
        return jsonify(settings.to_dict())


@template_bp.route('/download/<template_type>', methods=['GET'])
def download_template(template_type):
    """Download CSV/Excel template (stub)"""
    return jsonify({'message': 'Template download not yet implemented'}), 501

docs_bp = Blueprint('docs', __name__)

@docs_bp.route('', methods=['GET'])
def list_docs():
    """List all documentation files"""
    import os
    docs_dir = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), '..', 'z_docs')
    
    if not os.path.exists(docs_dir):
        return jsonify([])
        
    files = [f for f in os.listdir(docs_dir) if f.endswith('.md')]
    return jsonify(sorted(files))

@docs_bp.route('/<filename>', methods=['GET'])
def get_doc_content(filename):
    """Get content of a documentation file"""
    import os
    docs_dir = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), '..', 'z_docs')
    file_path = os.path.join(docs_dir, filename)
    
    if not os.path.exists(file_path):
        return jsonify({'error': 'File not found'}), 404
        
    with open(file_path, 'r') as f:
        content = f.read()
        
    return jsonify({'filename': filename, 'content': content})
