# IMPLEMENTATION OVERVIEW

## Architecture

### Backend (Node.js + Express)
- **API Server** (`backend/src/index.js`): Express.js HTTP server on port 3000
- **Database** (`backend/src/db.js`): SQLite3 with FTS5 virtual tables for full-text search
- **Workers** (`backend/src/workers/`): BullMQ job queue processing XML uploads asynchronously
- **Routes**: RESTful API for upload, search, bookmarks, tags, export

### Worker Process
- **XML Parser** (`backend/src/utils/xmlParser.js`): Streaming SAX parser (no full XML in memory)
- **Job Handler** (`backend/src/workers/xmlIndexer.js`): Processes BullMQ jobs
- **Progress Updates**: WebSocket via Redis pub/sub for real-time feedback

### Frontend (React + Vite)
- **SPA Router**: React Router v6 with nested routes
- **Vite Config**: Configured for subpath `/xml-tools/sms/` deployment
- **Components**:
  - Upload: Chunked file upload with progress
  - Search: Multi-mode search engine (exact, regex, fuzzy, relevance)
  - Results: Message display with context viewer
  - Dashboard: Central hub for history, saved searches, bookmarks, tags
  - Bookmarks: Searchable bookmark manager with tagging
  - Tags: Tag CRUD and organization

### Database Schema

#### Messages Table
```sql
CREATE TABLE messages (
  id TEXT PRIMARY KEY,
  thread_id TEXT,
  address TEXT,
  timestamp_ms INTEGER,
  direction TEXT,
  body TEXT,
  created_at DATETIME
);

CREATE INDEX idx_messages_thread_id ON messages(thread_id);
CREATE INDEX idx_messages_address ON messages(address);
CREATE INDEX idx_messages_timestamp ON messages(timestamp_ms);
```

#### FTS5 Index (Virtual Table)
```sql
CREATE VIRTUAL TABLE messages_fts USING fts5(
  id UNINDEXED,
  body,
  address UNINDEXED,
  content=messages,
  content_rowid=rowid
);
```

#### Related Tables
- `search_history`: Logs all searches (auto-cleanup after 1000 entries)
- `saved_searches`: User-defined named searches
- `bookmarks`: Marked messages with notes
- `tags`: Custom tag definitions with colors
- `bookmark_tags`: Junction table linking bookmarks to tags
- `uploads`: Track upload progress and status

### Search Modes

1. **Exact Match**
   ```sql
   SELECT * FROM messages WHERE body = ?
   ```

2. **Regex Match**
   ```sql
   SELECT * FROM messages WHERE body REGEXP ?
   ```

3. **Fuzzy Match**
   - Uses Levenshtein distance algorithm
   - Client-side filtering for flexibility

4. **Relevance (FTS5 Ranked)**
   ```sql
   SELECT * FROM messages_fts 
   WHERE messages_fts MATCH ?
   ORDER BY rank
   ```

### File Upload Flow

```
Browser
  ↓ (chunked POST)
Express Upload Handler
  ↓ (validates, stores in /uploads)
Enqueue Job → Redis
  ↓
Worker Process (detects new upload)
  ↓
Streaming XML Parser (SAX)
  ↓ (per message)
Batch Insert to SQLite (1000 at a time)
  ↓
Update FTS5 Indexes
  ↓
Mark upload complete
  ↓
Cleanup temp files
```

### Export Pipeline

```
POST /api/export/csv
  ↓
Execute search query
  ↓
Fetch all matching messages
  ↓
Format as CSV
  ↓
Stream response with attachment headers
```

---

## Configuration Files

### Backend `.env`
```
NODE_ENV=production
PORT=3000
HOST=127.0.0.1
DATABASE_PATH=/path/to/sms.db
UPLOAD_DIR=/path/to/uploads
REDIS_HOST=127.0.0.1
REDIS_PORT=6379
CORS_ORIGIN=https://code.jetlifecdn.com
MAX_UPLOAD_SIZE=1073741824
WORKER_CONCURRENCY=2
JOB_TIMEOUT=300000
```

### Frontend `.env`
```
VITE_API_URL=https://code.jetlifecdn.com/xml-tools/sms/api/
VITE_PUBLIC_PATH=/xml-tools/sms/
```

### Apache Configuration Snippet
```apache
ProxyPass /xml-tools/sms/api/ http://127.0.0.1:3000/ timeout=300
ProxyPassReverse /xml-tools/sms/api/ http://127.0.0.1:3000/
RewriteRule ^/xml-tools/sms/api/(.*)$ - [L]
RewriteRule . /xml-tools/sms/index.html [L]
```

### Systemd Services
- `sms-xml-backend.service`: Main API server
- `sms-xml-worker.service`: Background job worker

---

## Key Dependencies

### Backend
- `express`: Web framework
- `sqlite3` + `sqlite`: Database driver
- `bullmq`: Job queue
- `redis`: Cache and queue backend
- `busboy`: File upload streaming
- `winston`: Logging
- `joi`: Input validation

### Frontend
- `react`: UI framework
- `react-router-dom`: Client-side routing
- `axios`: HTTP client (or fetch)
- `tailwindcss`: Styling
- `date-fns`: Date formatting

---

## Performance Optimizations

1. **Streaming XML Parser**: Prevents loading entire file into memory
2. **SQLite WAL Mode**: Write-Ahead Logging for concurrent access
3. **FTS5 Virtual Tables**: Optimized full-text search indexes
4. **Batch Inserts**: 1000 messages per transaction
5. **Index on Common Filters**: address, timestamp, thread_id, direction
6. **Redis-Backed Job Queue**: Offload processing from HTTP handler
7. **Vite Code Splitting**: Dynamic imports for smaller initial bundle
8. **HTTP Caching**: ETag and Cache-Control headers

---

## Security Measures

1. **Input Validation**: Joi schemas for all endpoints
2. **SQL Injection Prevention**: Parameterized queries everywhere
3. **XSS Prevention**: HTML entity escaping, React default escaping
4. **CSRF**: SameSite cookies, CORS validation
5. **File Upload Security**: XML validation, size limits, type checking
6. **Filename Sanitization**: Prevent directory traversal
7. **Security Headers**: X-Content-Type-Options, X-Frame-Options, etc.
8. **Error Handling**: Generic error messages in production logs

---

## Monitoring & Health

- Health check endpoint: `GET /api/health`
- Structured logging with Winston
- Log rotation recommended
- Redis memory monitoring
- SQLite database integrity checks
- Uptime tracking in systemd

---

## Deployment

### Directory Structure After Installation
```
/mnt/media_drive2/site-root/code.jetlifecdn.com/xml-tools/sms/
├── backend/
│   ├── src/
│   │   ├── index.js
│   │   ├── db.js
│   │   ├── logger.js
│   │   ├── config.js
│   │   ├── api/
│   │   │   ├── routes.js
│   │   │   ├── upload.js
│   │   │   ├── search.js
│   │   │   ├── bookmarks.js
│   │   │   ├── searches.js
│   │   │   ├── tags.js
│   │   │   └── export.js
│   │   ├── workers/
│   │   │   ├── index.js
│   │   │   └── xmlIndexer.js
│   │   ├── utils/
│   │   │   ├── xmlParser.js
│   │   │   ├── validators.js
│   │   │   ├── sanitizers.js
│   │   │   └── csv.js
│   │   └── middleware/
│   │       ├── errorHandler.js
│   │       ├── auth.js
│   │       └── cors.js
│   ├── migrations/
│   │   ├── 001_init_schema.sql
│   │   └── 002_add_indexes.sql
│   ├── package.json
│   ├── .env
│   └── node_modules/
├── frontend/
│   ├── src/
│   │   ├── main.jsx
│   │   ├── App.jsx
│   │   ├── index.css
│   │   ├── components/
│   │   │   ├── Upload.jsx
│   │   │   ├── SearchEngine.jsx
│   │   │   ├── Results.jsx
│   │   │   ├── Dashboard.jsx
│   │   │   ├── Bookmarks.jsx
│   │   │   ├── Tags.jsx
│   │   │   ├── SavedSearches.jsx
│   │   │   └── SearchHistory.jsx
│   │   ├── pages/
│   │   │   ├── HomePage.jsx
│   │   │   ├── DashboardPage.jsx
│   │   │   └── SettingsPage.jsx
│   │   ├── hooks/
│   │   │   ├── useApi.js
│   │   │   ├── useSearch.js
│   │   │   ├── useBookmarks.js
│   │   │   └── useTags.js
│   │   └── utils/
│   │       ├── api.js
│   │       └── formatters.js
│   ├── vite.config.js
│   ├── index.html
│   ├── package.json
│   ├── .env
│   └── node_modules/
├── databases/
│   └── sms.db
│       └── sms.db-shm (WAL files)
│       └── sms.db-wal
├── uploads/
│   └── [temporary upload files]
├── logs/
│   ├── backend.log
│   ├── worker.log
│   └── access.log
├── install_sms_xml_tool.sh
└── README.md
```

### Installation Steps Summary
1. Extract ZIP to temporary location
2. Run `sudo ./install_sms_xml_tool.sh` as root
3. Script creates directories, installs deps, initializes DB
4. Update Apache vhost config with proxy rules
5. Enable Apache modules: `proxy`, `proxy_http`, `proxy_wstunnel`, `rewrite`
6. Start services: `systemctl start sms-xml-backend sms-xml-worker`
7. Access at: `https://code.jetlifecdn.com/xml-tools/sms/`

---

## Assumptions

1. **User Network**: Application assumes trusted/internal network (optional: add JWT auth)
2. **SMS XML Format**: Standard Android SMS backup XML format with message structure
3. **Database Scope**: Single SQLite database (no multi-instance support needed)
4. **Concurrent Users**: Single-digit concurrent users (scaling: add SQLite replicas or PostgreSQL)
5. **File Size**: Up to 1GB (>1GB would require sharding or streaming export)
6. **Apache Setup**: Assuming existing Apache2 vhosts with SSL already configured
7. **Redis Local**: Assuming Redis on same machine (localhost:6379)
8. **Disk Space**: 3x file size (upload + DB + backups)

---

## Future Enhancements

- [ ] JWT authentication with user accounts
- [ ] Multi-file simultaneous uploads
- [ ] Advanced analytics dashboard
- [ ] Bulk tagging/bookmarking
- [ ] Message thread viewer
- [ ] Export to multiple formats (JSON, Parquet)
- [ ] Full-text search refinements (operators, wildcards)
- [ ] Mobile-responsive UI improvements
- [ ] PostgreSQL backend option
- [ ] API rate limiting
- [ ] Message attachment support
- [ ] Conversation threading UI
