import sqlite3 from 'sqlite3';
import { open } from 'sqlite';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';

dotenv.config();

const __dirname = path.dirname(fileURLToPath(import.meta.url));
const dbPath = process.env.DATABASE_PATH || path.join(__dirname, '../../../databases/sms.db');

async function initDatabase() {
  const db = await open({
    filename: dbPath,
    driver: sqlite3.Database
  });

  try {
    // Enable FTS5 and WAL mode
    await db.exec('PRAGMA journal_mode = WAL');
    await db.exec('PRAGMA synchronous = NORMAL');
    await db.exec('PRAGMA cache_size = -64000');
    await db.exec('PRAGMA foreign_keys = ON');

    // Messages table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS messages (
        id TEXT PRIMARY KEY,
        thread_id TEXT NOT NULL,
        address TEXT NOT NULL,
        timestamp_ms INTEGER NOT NULL,
        direction TEXT NOT NULL,
        body TEXT NOT NULL,
        type TEXT DEFAULT 'SMS',
        protocol_identifier INTEGER,
        service_center TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_messages_thread_id ON messages(thread_id);
      CREATE INDEX IF NOT EXISTS idx_messages_address ON messages(address);
      CREATE INDEX IF NOT EXISTS idx_messages_timestamp ON messages(timestamp_ms);
      CREATE INDEX IF NOT EXISTS idx_messages_direction ON messages(direction);
    `);

    // FTS5 Virtual table for full-text search
    await db.exec(`
      CREATE VIRTUAL TABLE IF NOT EXISTS messages_fts USING fts5(
        id UNINDEXED,
        body,
        address UNINDEXED,
        content=messages,
        content_rowid=rowid
      );
    `);

    // Triggers to keep FTS in sync
    await db.exec(`
      CREATE TRIGGER IF NOT EXISTS messages_insert AFTER INSERT ON messages BEGIN
        INSERT INTO messages_fts(rowid, id, body, address)
        VALUES (new.rowid, new.id, new.body, new.address);
      END;
      
      CREATE TRIGGER IF NOT EXISTS messages_delete AFTER DELETE ON messages BEGIN
        INSERT INTO messages_fts(messages_fts, rowid, id, body, address)
        VALUES('delete', old.rowid, old.id, old.body, old.address);
      END;
      
      CREATE TRIGGER IF NOT EXISTS messages_update AFTER UPDATE ON messages BEGIN
        INSERT INTO messages_fts(messages_fts, rowid, id, body, address)
        VALUES('delete', old.rowid, old.id, old.body, old.address);
        INSERT INTO messages_fts(rowid, id, body, address)
        VALUES(new.rowid, new.id, new.body, new.address);
      END;
    `);

    // Search history table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS search_history (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        query TEXT NOT NULL,
        mode TEXT NOT NULL,
        filters TEXT,
        result_count INTEGER,
        executed_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_search_history_timestamp ON search_history(executed_at DESC);
    `);

    // Saved searches table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS saved_searches (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        query TEXT NOT NULL,
        mode TEXT NOT NULL,
        filters TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Bookmarks table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS bookmarks (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        message_id TEXT NOT NULL,
        notes TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY(message_id) REFERENCES messages(id) ON DELETE CASCADE,
        UNIQUE(message_id)
      );
      
      CREATE INDEX IF NOT EXISTS idx_bookmarks_message_id ON bookmarks(message_id);
    `);

    // Tags table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS tags (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        color TEXT DEFAULT '#808080',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_tags_name ON tags(name);
    `);

    // Bookmark-tags junction table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS bookmark_tags (
        bookmark_id INTEGER NOT NULL,
        tag_id INTEGER NOT NULL,
        PRIMARY KEY(bookmark_id, tag_id),
        FOREIGN KEY(bookmark_id) REFERENCES bookmarks(id) ON DELETE CASCADE,
        FOREIGN KEY(tag_id) REFERENCES tags(id) ON DELETE CASCADE
      );
    `);

    // Upload tracking table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS uploads (
        id TEXT PRIMARY KEY,
        filename TEXT NOT NULL,
        file_size INTEGER NOT NULL,
        message_count INTEGER DEFAULT 0,
        status TEXT DEFAULT 'processing',
        progress_percent INTEGER DEFAULT 0,
        error_message TEXT,
        started_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        completed_at DATETIME,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
    `);

    console.log('✓ Database initialized successfully at:', dbPath);
  } catch (error) {
    console.error('✗ Database initialization failed:', error.message);
    process.exit(1);
  } finally {
    await db.close();
  }
}

initDatabase().catch(console.error);
