#!/bin/bash
#
# SMS XML Search & Bookmarking Tool - Installation Script
# Target: Ubuntu Server 24.04 with Apache2
# Installs to: /mnt/media_drive2/site-root/code.jetlifecdn.com/xml-tools/sms/
#

set -e

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
APP_ROOT="/mnt/media_drive2/site-root/code.jetlifecdn.com/xml-tools/sms"
APP_USER="www-data"
APP_GROUP="www-data"
NODE_VERSION="20"
REDIS_PORT="6379"
API_PORT="3000"
VHOST_DOMAIN="code.jetlifecdn.com"
PUBLIC_URL="https://code.jetlifecdn.com/xml-tools/sms/"

echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║  SMS XML Search & Bookmarking Tool - Installation${NC}"
echo -e "${BLUE}║  Target: ${VHOST_DOMAIN}/xml-tools/sms/${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}Error: This script must be run as root${NC}"
   exit 1
fi

# Check for required tools
echo -e "${YELLOW}[1/8] Checking prerequisites...${NC}"
command -v node &> /dev/null || { echo -e "${RED}Node.js is not installed${NC}"; exit 1; }
command -v npm &> /dev/null || { echo -e "${RED}npm is not installed${NC}"; exit 1; }
command -v git &> /dev/null || { echo -e "${RED}git is not installed${NC}"; exit 1; }

NODE_VERSION_INSTALLED=$(node -v)
echo -e "${GREEN}✓ Node.js ${NODE_VERSION_INSTALLED}${NC}"
echo -e "${GREEN}✓ npm $(npm -v)${NC}"

# Check Apache
echo -e "${YELLOW}[2/8] Checking Apache2 configuration...${NC}"
if ! systemctl is-active --quiet apache2; then
    echo -e "${YELLOW}⚠ Apache2 is not running. Starting...${NC}"
    systemctl start apache2
fi
echo -e "${GREEN}✓ Apache2 running${NC}"

# Check if vhost exists
if ! ls /etc/apache2/sites-available/code.jetlifecdn.com* &> /dev/null; then
    echo -e "${RED}Error: Apache vhost files not found at /etc/apache2/sites-available/code.jetlifecdn.com*.conf${NC}"
    exit 1
fi
echo -e "${GREEN}✓ Apache vhost configuration found${NC}"

# Create application directory structure
echo -e "${YELLOW}[3/8] Creating directory structure...${NC}"
mkdir -p "${APP_ROOT}"/{backend,frontend,uploads,databases,logs}
mkdir -p "${APP_ROOT}/backend"/{src,migrations,node_modules,scripts}
mkdir -p "${APP_ROOT}/backend/src"/{api,workers,utils,middleware}
mkdir -p "${APP_ROOT}/frontend"/{src,dist,node_modules}
mkdir -p "${APP_ROOT}/frontend/src"/{components,pages,hooks,utils}

echo -e "${GREEN}✓ Directories created${NC}"

# Set permissions
echo -e "${YELLOW}[4/8] Setting permissions...${NC}"
chown -R ${APP_USER}:${APP_GROUP} "${APP_ROOT}"
chmod -R 755 "${APP_ROOT}"
chmod -R 775 "${APP_ROOT}"/{uploads,databases,logs}
find "${APP_ROOT}" -type d -exec chmod 755 {} \;
find "${APP_ROOT}"/{uploads,databases,logs} -type d -exec chmod 775 {} \;

echo -e "${GREEN}✓ Permissions configured${NC}"

# Install backend dependencies
echo -e "${YELLOW}[5/8] Installing backend dependencies...${NC}"
cd "${APP_ROOT}/backend"

# Copy package.json from script directory if available, otherwise create minimal
if [ ! -f "package.json" ]; then
    cat > package.json << 'EOF'
{
  "name": "sms-xml-search-backend",
  "version": "1.0.0",
  "type": "module",
  "main": "src/index.js",
  "scripts": {
    "start": "node src/index.js",
    "dev": "NODE_ENV=development node src/index.js",
    "worker": "node src/workers/index.js",
    "init-db": "node src/scripts/initDb.js"
  },
  "dependencies": {
    "express": "^4.18.2",
    "cors": "^2.8.5",
    "dotenv": "^16.3.1",
    "sqlite3": "^5.1.6",
    "sqlite": "^5.0.1",
    "busboy": "^1.4.0",
    "bullmq": "^5.0.0",
    "redis": "^4.6.11",
    "winston": "^3.11.0",
    "joi": "^17.11.0",
    "papaparse": "^5.4.1"
  },
  "devDependencies": {
    "nodemon": "^3.0.1"
  }
}
EOF
fi

npm install
echo -e "${GREEN}✓ Backend dependencies installed${NC}"

# Create environment file
echo -e "${YELLOW}[6/8] Configuring environment...${NC}"
if [ ! -f ".env" ]; then
    cat > .env << EOF
NODE_ENV=production
PORT=${API_PORT}
HOST=127.0.0.1
LOG_LEVEL=info
LOG_FILE=${APP_ROOT}/logs/backend.log

# Database
DATABASE_PATH=${APP_ROOT}/databases/sms.db
MAX_UPLOAD_SIZE=1073741824
UPLOAD_DIR=${APP_ROOT}/uploads

# Redis (BullMQ for background jobs)
REDIS_HOST=127.0.0.1
REDIS_PORT=${REDIS_PORT}

# Security
CORS_ORIGIN=https://${VHOST_DOMAIN}
SESSION_SECRET=$(openssl rand -hex 32)

# Public URLs
PUBLIC_URL=${PUBLIC_URL}
API_URL=${PUBLIC_URL}api/

# Worker settings
WORKER_CONCURRENCY=2
JOB_TIMEOUT=300000
EOF
    chmod 600 .env
    chown ${APP_USER}:${APP_GROUP} .env
    echo -e "${GREEN}✓ Backend .env created${NC}"
else
    echo -e "${GREEN}✓ Backend .env already exists${NC}"
fi

# Create database initialization script in correct location
echo -e "${YELLOW}[7/8] Initializing SQLite database...${NC}"
mkdir -p "${APP_ROOT}/backend/src/scripts"

cat > "${APP_ROOT}/backend/src/scripts/initDb.js" << 'INITDB'
import sqlite3 from 'sqlite3';
import { open } from 'sqlite';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';

dotenv.config();

const __dirname = path.dirname(fileURLToPath(import.meta.url));
const dbPath = process.env.DATABASE_PATH || path.join(__dirname, '../../../databases/sms.db');

async function initDatabase() {
  const db = await open({
    filename: dbPath,
    driver: sqlite3.Database
  });

  try {
    // Enable FTS5 and WAL mode
    await db.exec('PRAGMA journal_mode = WAL');
    await db.exec('PRAGMA synchronous = NORMAL');
    await db.exec('PRAGMA cache_size = -64000');
    await db.exec('PRAGMA foreign_keys = ON');

    // Messages table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS messages (
        id TEXT PRIMARY KEY,
        thread_id TEXT NOT NULL,
        address TEXT NOT NULL,
        timestamp_ms INTEGER NOT NULL,
        direction TEXT NOT NULL,
        body TEXT NOT NULL,
        type TEXT DEFAULT 'SMS',
        protocol_identifier INTEGER,
        service_center TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_messages_thread_id ON messages(thread_id);
      CREATE INDEX IF NOT EXISTS idx_messages_address ON messages(address);
      CREATE INDEX IF NOT EXISTS idx_messages_timestamp ON messages(timestamp_ms);
      CREATE INDEX IF NOT EXISTS idx_messages_direction ON messages(direction);
    `);

    // FTS5 Virtual table for full-text search
    await db.exec(`
      CREATE VIRTUAL TABLE IF NOT EXISTS messages_fts USING fts5(
        id UNINDEXED,
        body,
        address UNINDEXED,
        content=messages,
        content_rowid=rowid
      );
    `);

    // Triggers to keep FTS in sync
    await db.exec(`
      CREATE TRIGGER IF NOT EXISTS messages_insert AFTER INSERT ON messages BEGIN
        INSERT INTO messages_fts(rowid, id, body, address)
        VALUES (new.rowid, new.id, new.body, new.address);
      END;
      
      CREATE TRIGGER IF NOT EXISTS messages_delete AFTER DELETE ON messages BEGIN
        INSERT INTO messages_fts(messages_fts, rowid, id, body, address)
        VALUES('delete', old.rowid, old.id, old.body, old.address);
      END;
      
      CREATE TRIGGER IF NOT EXISTS messages_update AFTER UPDATE ON messages BEGIN
        INSERT INTO messages_fts(messages_fts, rowid, id, body, address)
        VALUES('delete', old.rowid, old.id, old.body, old.address);
        INSERT INTO messages_fts(rowid, id, body, address)
        VALUES(new.rowid, new.id, new.body, new.address);
      END;
    `);

    // Search history table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS search_history (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        query TEXT NOT NULL,
        mode TEXT NOT NULL,
        filters TEXT,
        result_count INTEGER,
        executed_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_search_history_timestamp ON search_history(executed_at DESC);
    `);

    // Saved searches table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS saved_searches (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        query TEXT NOT NULL,
        mode TEXT NOT NULL,
        filters TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Bookmarks table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS bookmarks (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        message_id TEXT NOT NULL,
        notes TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY(message_id) REFERENCES messages(id) ON DELETE CASCADE,
        UNIQUE(message_id)
      );
      
      CREATE INDEX IF NOT EXISTS idx_bookmarks_message_id ON bookmarks(message_id);
    `);

    // Tags table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS tags (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        color TEXT DEFAULT '#808080',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_tags_name ON tags(name);
    `);

    // Bookmark-tags junction table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS bookmark_tags (
        bookmark_id INTEGER NOT NULL,
        tag_id INTEGER NOT NULL,
        PRIMARY KEY(bookmark_id, tag_id),
        FOREIGN KEY(bookmark_id) REFERENCES bookmarks(id) ON DELETE CASCADE,
        FOREIGN KEY(tag_id) REFERENCES tags(id) ON DELETE CASCADE
      );
    `);

    // Upload tracking table
    await db.exec(`
      CREATE TABLE IF NOT EXISTS uploads (
        id TEXT PRIMARY KEY,
        filename TEXT NOT NULL,
        file_size INTEGER NOT NULL,
        message_count INTEGER DEFAULT 0,
        status TEXT DEFAULT 'processing',
        progress_percent INTEGER DEFAULT 0,
        error_message TEXT,
        started_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        completed_at DATETIME,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
    `);

    console.log('✓ Database initialized successfully at:', dbPath);
  } catch (error) {
    console.error('✗ Database initialization failed:', error.message);
    process.exit(1);
  } finally {
    await db.close();
  }
}

initDatabase().catch(console.error);
INITDB

chown ${APP_USER}:${APP_GROUP} "${APP_ROOT}/backend/src/scripts/initDb.js"
chmod 755 "${APP_ROOT}/backend/src/scripts/initDb.js"

# Run database initialization with correct path
cd "${APP_ROOT}/backend"
sudo -u ${APP_USER} bash -c "export NODE_PATH=${APP_ROOT}/backend/node_modules && node src/scripts/initDb.js"

echo -e "${GREEN}✓ SQLite database initialized${NC}"

# Create systemd service for Node.js backend
echo -e "${YELLOW}[8/8] Creating systemd services...${NC}"
cat > /etc/systemd/system/sms-xml-backend.service << EOF
[Unit]
Description=SMS XML Search Backend API
After=network.target redis-server.service
Wants=redis-server.service

[Service]
Type=simple
User=${APP_USER}
Group=${APP_GROUP}
WorkingDirectory=${APP_ROOT}/backend
ExecStart=/usr/bin/node src/index.js
Restart=on-failure
RestartSec=5
StandardOutput=append:${APP_ROOT}/logs/backend.log
StandardError=append:${APP_ROOT}/logs/backend.log
Environment="NODE_ENV=production"
Environment="NODE_OPTIONS=--max-old-space-size=2048"

[Install]
WantedBy=multi-user.target
EOF

cat > /etc/systemd/system/sms-xml-worker.service << EOF
[Unit]
Description=SMS XML Search Background Worker
After=network.target redis-server.service
Wants=redis-server.service sms-xml-backend.service

[Service]
Type=simple
User=${APP_USER}
Group=${APP_GROUP}
WorkingDirectory=${APP_ROOT}/backend
ExecStart=/usr/bin/node src/workers/index.js
Restart=on-failure
RestartSec=5
StandardOutput=append:${APP_ROOT}/logs/worker.log
StandardError=append:${APP_ROOT}/logs/worker.log
Environment="NODE_ENV=production"
Environment="NODE_OPTIONS=--max-old-space-size=2048"

[Install]
WantedBy=multi-user.target
EOF

systemctl daemon-reload
echo -e "${GREEN}✓ Systemd services created${NC}"

# Configure Apache
echo -e "${BLUE}────────────────────────────────────────────────────────────${NC}"
echo -e "${YELLOW}Apache Configuration Required${NC}"
echo -e "${BLUE}────────────────────────────────────────────────────────────${NC}"
echo ""
echo "Add this snippet to your Apache vhost config at:"
echo "  /etc/apache2/sites-available/code.jetlifecdn.com.conf"
echo "  /etc/apache2/sites-available/code.jetlifecdn.com-le-ssl.conf"
echo ""
echo "Inside the <VirtualHost> block, add:"
echo ""
cat << 'APACHE'
    # SMS XML Search Tool
    <Directory /mnt/media_drive2/site-root/code.jetlifecdn.com/xml-tools/sms>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
        
        # SPA routing - rewrite all requests to index.html
        <IfModule mod_rewrite.c>
            RewriteEngine On
            RewriteBase /xml-tools/sms/
            RewriteCond %{REQUEST_FILENAME} !-f
            RewriteCond %{REQUEST_FILENAME} !-d
            RewriteRule ^api/(.*)$ - [L]
            RewriteRule . /xml-tools/sms/index.html [L]
        </IfModule>
    </Directory>
    
    # Reverse proxy for API calls
    ProxyPreserveHost On
    ProxyPass /xml-tools/sms/api/ http://127.0.0.1:3000/ timeout=300
    ProxyPassReverse /xml-tools/sms/api/ http://127.0.0.1:3000/
    
    # WebSocket support for progress updates
    RewriteEngine On
    RewriteCond %{HTTP:Upgrade} websocket [NC]
    RewriteCond %{HTTP:Connection} upgrade [NC]
    RewriteRule ^/xml-tools/sms/api/(.*)$ "ws://127.0.0.1:3000/$1" [P,L]
APACHE

echo ""
echo "Then run:"
echo "  sudo a2enmod proxy"
echo "  sudo a2enmod proxy_http"
echo "  sudo a2enmod proxy_wstunnel"
echo "  sudo a2enmod rewrite"
echo "  sudo a2enmod headers"
echo "  sudo apache2ctl configtest"
echo "  sudo systemctl reload apache2"
echo ""

# Check if Redis is installed
if ! command -v redis-server &> /dev/null; then
    echo -e "${YELLOW}⚠ Redis not found. Installing...${NC}"
    apt-get update -qq
    apt-get install -y redis-server
    systemctl start redis-server
    systemctl enable redis-server
    echo -e "${GREEN}✓ Redis installed and enabled${NC}"
else
    echo -e "${GREEN}✓ Redis already installed${NC}"
    if ! systemctl is-active --quiet redis-server; then
        systemctl start redis-server
        echo -e "${GREEN}✓ Redis started${NC}"
    fi
fi

# Print final status
echo ""
echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║  Installation Complete${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${GREEN}Next Steps:${NC}"
echo ""
echo "1. ${YELLOW}Update Apache vhost configuration:${NC}"
echo "   Add proxy rules and rewrite rules (see above)"
echo "   sudo nano /etc/apache2/sites-available/code.jetlifecdn.com.conf"
echo "   sudo nano /etc/apache2/sites-available/code.jetlifecdn.com-le-ssl.conf"
echo ""
echo "2. ${YELLOW}Enable required Apache modules:${NC}"
echo "   sudo a2enmod proxy proxy_http proxy_wstunnel rewrite headers"
echo ""
echo "3. ${YELLOW}Test Apache configuration:${NC}"
echo "   sudo apache2ctl configtest"
echo ""
echo "4. ${YELLOW}Reload Apache:${NC}"
echo "   sudo systemctl reload apache2"
echo ""
echo "5. ${YELLOW}Start services:${NC}"
echo "   sudo systemctl start sms-xml-backend"
echo "   sudo systemctl start sms-xml-worker"
echo ""
echo "6. ${YELLOW}Check service status:${NC}"
echo "   sudo systemctl status sms-xml-backend"
echo "   sudo systemctl status sms-xml-worker"
echo ""
echo "7. ${YELLOW}View logs:${NC}"
echo "   tail -f ${APP_ROOT}/logs/backend.log"
echo "   tail -f ${APP_ROOT}/logs/worker.log"
echo ""
echo -e "${GREEN}Application URL:${NC} ${PUBLIC_URL}"
echo -e "${GREEN}API Endpoint:${NC} ${PUBLIC_URL}api/"
echo ""
echo -e "${YELLOW}Default credentials:${NC} See .env file"
echo ""
